/*
  Manages each player for Bombing Run. Similar to UT Controller class.
  Has a good portion of the Bot AI.
  
  Author Mark Caldwell aka W@rHe@d of The Reliquary
*/
 
/*
pathing theory

ut has been found to be horrible with pathing so for BR we have our own special
pathing. part of that is dynamic pathing, which is called in this code 
Special Navigation. Special navigation allows a bot to dynamically seek
a target while navigating around blockages. lots of tracing is done so it's
expensive and therefore we use special navigation only when needed.
we also have fallback algorithms for when pathing fails.

the most common fallback is the DefensivePosition. if pathing fails then bot
can seek a pathable navigation point near to the objective and then from there
try to find objective.

the path cycle:
-FindPathToObjective
-If path fails, perhaps use Special navigation to objective
-If still fail, seek DefensivePosition
-If still fail, or if reached DefensivePosition, or near enough
 to objective, use Special Navigation to objective
-If still fail, perhaps find a random destination for a few seconds
-Go back to top of cycle

Defensive Positions
A random navigation point in a near radius to objective is selected.
closer is not always better. we do not try to select the
closest DefensivePosition to the target thinking that will
be the best place to reach target from. picture a goal against
a wall in a building. on other side of wall, two feet away, 
is a path node on a ledge outside of building. that path node
is the closest node but goal is totally unreachable from that 
node. therefore when selecting a DefensivePosition we try to
select from a reasonably wide radius of nodes which are vertically
and horizontally close to target to have better chance of selecting 
a good one. 

picture a map with just a floor and a goal, nothing else, in
regular gravity.
if the goal was on the floor and you were 10 miles away, you
can reach the goal by walking.
if the goal is just 10 feet off the floor you can never reach the goal.

this is why special navigation is more feasible horizontally than vertically.

we allow special navigation if the bot is reasonably close to
the target and can jump to it's height. special navigation will 
walk bot to target and jump.

if bot can't jump the needed height then it's prefered to try a DefensivePosition
which is closer to target vertically.
*/

class UTBRPlayerManager extends Actor;

const CAN_SEE = 1;
const SHOOT_GOAL = 2;
const CLEAR_SHOT_PATH = 3;
const SHOOT_BALL = 4;
const SHOOT_RESET = 5;
const CLEAR_BALL_RESET_KILLZ_PATH = 6;

const PMTimerInterval = 0.25;

struct RouteHistoryNode
{
    var Actor Target;
    var bool Reached;
    var bool AllowRevisit;
};

var UTBRGame BRGame;
var UTBRSquadAI Squad;
var Controller Controller;
var UTBot Bot;
var RepNotify PlayerReplicationInfo PlayerReplicationInfo;
var int SpectatorPriIndex;
var int Kills;
var	int	MaxMultiJump;
var Actor SpectatorViewTarget;
var PlayerReplicationInfo SpectatorViewTargetPRI;
var bool SpectatorFreeCamera; 
var Rotator CameraLastRotation;
var float CameraRotationOffset;
var Actor LastViewTarget;
var Rotator LastViewTargetRotation;
var vector CameraOffset, CameraLastLocation;
var vector PawnLocation;
var Rotator PawnRotation;
var bool PawnVisibleOnMiniMap;
var float BotAiTime;
var float OldHealth;
var float CurrentHealth;
var float BotShootTime;
var bool WaitingForMultiJumpApex;
var float LastZ;
var bool ShootingGoal;
var bool SpecialNavigating;
var bool NoGoalPath;
var bool NoBallPath;
var bool TryJumpShot;
var Actor MoveTarget;
var bool KeepJumping;
var bool HeadingTowardsEnemyGoal;
var float oldImpactJumpZ;
var bool TurningForGoalJumpShot;
var bool DoingBallReset;
var bool AttackBallBoosting;
var bool DefenseBallBoosting;
var float DefensivePositionTime;
var float TakingDefensivePathTime;
var bool CampAtDefensivePosition;
var bool ForceCamp;
var UTBRBallLauncher BallLauncher;
var UTBRBall TheBall;
var Actor LastFormationCenter;
var float DistFromMoveTarget;
var float MoveTargetTime;
var float LastJumpTime;
var bool GoalJumping;
var PlayerController LocalPlayerController;
var float ResetBallTime;
var Actor LastMoveTarget;
var float GoodFloorDist;
var bool TriedPath;
var float DistFromDesiredPosition;
var vector GoalJumpPosition;
var Vector JumpStartLocation, LastJumpStartLocation;
var float JumpHeight;
var bool ForceDefensivePosition;
var Actor FormationCenter;
var Vector FormationCenterLocation;
var float HorizontalDistanceFromFormationCenter;
var float HorizontalDistanceFromEnemyGoal;
var float DistanceFromFormationCenter;
var float DistanceFromEnemyGoal;
var float MaxFormationCenterDefenseDistance;
var float MaxFriendlyGoalDefenseDistance;
var float VerticalDistanceFromFormationCenter;
var bool CanJumpToFormationCenter;
var bool CanJumpToEnemyGoalHeight;
var bool CanJumpToFormationCenterHeight;
var bool CanJumpToEnemyGoal;
var float JumpApexHorizontalDistance;
var float BallResetDist;
var float KillZResetDist;
var float VerticalDistanceFromEnemyGoal;
var float HorizontalDistanceFromFriendlyGoal;
var Vector CheckFormationCenterLocation;
var bool SeekDirect;
var bool CanResetBall;
var float LobPassDistance;
var bool GettingWeapon;
var bool GettingVehicle;
var bool GettingSuperPickup;
var bool Blocked;
var bool ZBlocked;
var Vector StrafeDirection;
var bool GoalJumpingWithBall;
var UTBRGoal GoalTarget;
var bool SpecialFocus;
var vector StrafeTarget;
var bool ZStrafing;
var float StrafeTime;
var float BlockedZ;
var int ZDirMult;
var bool HorizontallyNearTarget;
var Actor JumpStartTarget;
var Actor PathTarget;
var Vector BallSoakLocation;
var float BallSoakTime;
var float SoakTime;
var float SoakDist, SoakDistVert;
var Vector SoakLocation;
var bool CheckSoak;
var bool ReachedFallback;
var bool OldHasTranslocator;
var float ZSurfaceDist;  //distance up to which good floor or ceiling found
var float ZSurfaceBreakDist;  //distance at which a hole/droppoff in floor or ceiling found
var Vector BlockedPawnLocation;
var Vector BlockedDirection;
var float BlockedDist;
var float zBlockDist;
var vector LastDestination;
var float SpecialNavigationTime;
var Actor SpecialNavigationFormationCenter;
var vector LastFormationCenterLocation;
var float LastDistanceFromFormationCenter;
var float ClosestDistanceToFormationCenter;
var Array<RouteHistoryNode> RouteHistory;
var Actor FormationCenterTarget;
var bool Steer;
var float SteerTime;
var float SteerDuration;
var UTVehicle LastVehicle;
var UTVehicle VehicleGoal;
var Actor FailedTarget;
var float SuperPickupTime;
var float SoakTargetDist;
var Actor SoakTarget;
var string Reason;
var float MaxSoakTime;
var bool FindingRandomTarget;
var EPhysics SpecialNavigationPhysics;
var bool DefensivePositionFailed;
var bool SoakAllowBlocking;
var float SoakBlockingDist, SoakBlockingDistVert;
var float SoakStrafeTargetDist;
var float SuppressEnemyTime;
var bool SuppressEnemy;
var float SuppressBoostTime;
var float WeaponDisallowBallPickupTime;
var Vector SoakTargetLocation;
var float VehicleSpecialNavigationTime;
var float BlockageTime;
var float FallbackPerimeterDistance;
var float SuppressDefensivePositionResetZ;
var bool ForceJump;
var bool SuppressLoopChecks;
var bool ForceRepath;
var bool LandedOnBall;
var Actor SpecialNavigationPathTarget;
var float LastRetaskTime;
var bool Retasking;
var vector BlockingStartLoc;
var float LobBlockedDist;
var float LobLandingCheckedDist;
var float LobLandingHeight;
var Vector LobLandingPoint;
var Actor CachedDefensivePosition;
var float EnemyLoseFocusTime;
var Vector AimFocalPoint;
var float AimTime;
var bool SpecialAiming;
var float FormationCenterMovedTime;
var bool JumpOverBlockage, JumpingOverBlockage;
var Vector Destination;
var bool JumpStartedByPlayerManager;
var Actor SoakSpecialNavigationTarget;
var bool TriedDefensivePosition;
var Actor FailPathTo;
var Vector RandomDestination;
var bool TriedForcedSpecialNavigationToPathTarget;
var Actor PathSubTarget;
var RouteInfoStruct BRNodeRouteInfo;
var Actor DefensiveCenterTarget;
var RouteInfoStruct NewMoveTargetInfo;
var bool CanUseAssaultPaths;
var bool IsSoaking;
var bool ForcedJumpInSoak;
var bool BackingUp;
var float LastYaw;
var float RotatingBigVehicleTime;
var float CollisionRadius;
var float CollisionHeight;
var bool UseBRMultiJump;
var bool UseBRMultiJumpBoost;
var UTBRGameSettingsScene SettingsScreen;


//return true if controller should be allowed access to admin function
simulated function bool AllowAdminAccess()
{
    local UTPlayerReplicationInfo pri;
 
    if (Controller == none)
    {
        return false;
    }
    
    if ((BRGame != none) && BRGame.AdminRightsNotNeeded)
    {
        return true;
    }
  
    if (Controller.WorldInfo.NetMode == NM_Standalone)
    {
        return true;
    }
   
    pri =  UTPlayerReplicationInfo(Controller.PlayerReplicationInfo);
    if (pri == none)
    {
        return false;
    }
   
    if (pri.bAdmin)
    {
        return true;
    }

    return false;
}

function string GetSettingsName(string map)
{
    if (map == "")
    {
        map = "Bombing Run Settings";
    }
    
    return map;
}

//alerts everyone someone has gone into settings screen while Online
reliable server function SendSettingsScreenAlert()
{
    BRGame.SendDebugMessage(Controller.PlayerReplicationInfo.GetPlayerAlias() $ " has gone into the BR Settings Screen");
}

//load maps for client settings screen
reliable server function LoadSettingsScreenMaps()
{
    local UTBRGameSettingsScene screen;
    local int i;

    if (! AllowAdminAccess())
    {
        return;
    }

    screen = new (none) class'BombingRun.UTBRGameSettingsScene';

    screen.GetMapList();
   
    //UT won't replicate an array nor will it send a large string so we must pass one at a time :(

    for (i = 0; i < screen.MapList.Length; i++)
    {    
        ClientLoadSettingsScreenMap(screen.MapList[i]);
    }
    ClientSetSettingsScreenMaps();
}

//update client settings screen with one map from server list
reliable client function ClientLoadSettingsScreenMap(string map)
{
     SettingsScreen.MapList[SettingsScreen.MapList.Length] = map;
}

//set client settings screen map combo box
reliable client function ClientSetSettingsScreenMaps()
{
     SettingsScreen.SetMaps();
}

//load mutators for client settings screen
reliable server function LoadSettingsScreenMutators()
{
    local UTBRGameSettingsScene screen;
    local int i;

    if (! AllowAdminAccess())
    {
        return;
    }

    screen = new (none) class'BombingRun.UTBRGameSettingsScene';

    screen.GetMutatorList();
    
  
    //UT won't replicate an array nor will it send a large string so we must pass one at a time :(

    for (i = 0; i < screen.MutatorNameList.Length; i++)
    {    
        ClientLoadSettingsScreenMutator(screen.MutatorNameList[i], screen.MutatorClassList[i]);
    }
    ClientSetSettingsScreenMutators();  
}

//update client with one mutator from server list
reliable client function ClientLoadSettingsScreenMutator(string mutName, string mutClassName)
{
     SettingsScreen.MutatorNameList[SettingsScreen.MutatorNameList.Length] = mutName;
     SettingsScreen.MutatorClassList[SettingsScreen.MutatorClassList.Length] = mutClassName;     
}

//set client settings screen mutator combo box
reliable client function ClientSetSettingsScreenMutators()
{
     SettingsScreen.SetMutators();
}

//get settings for client settings screen
reliable server function LoadSettingsScreenSettings(string map)
{
    local UTBRSettings settings;
       
    if (! AllowAdminAccess())
    {
        return;
    }
       
    settings = new (none, GetSettingsName(map)) class'BombingRun.UTBRSettings';   
    ClientLoadSettingsScreenSettings(map, settings.Serialize(), settings.mutators);    
}

//update settings to client settings screen
reliable client function ClientLoadSettingsScreenSettings(string map, string serializedData, string mutators)
{   
    SettingsScreen.Settings = new (none, GetSettingsName(map)) class'BombingRun.UTBRSettings';
    SettingsScreen.Settings.DeSerialize(serializedData, "");
    SettingsScreen.Settings.Mutators = mutators;
    SettingsScreen.SetFields();
}

//called to get game settings from server 
reliable server function GetServerGameSettings()
{
    UpdateClientGameSettings(BRGame.Settings.Serialize());
}

//called by GetServerGameSettings() to update server settings to client game
reliable client function UpdateClientGameSettings(string serializedData)
{
    BRGame.Settings = new (none, GetSettingsName("")) class'BombingRun.UTBRSettings';
    BRGame.Settings.DeSerialize(serializedData, "");   
   
    BRGame.ClientApplySettings();
}

//called when Restart Game clicked in settings screen
reliable server function RestartGame()
{
    if (! AllowAdminAccess())
    {
        return;
    }

    if (BRGame != none)
    {    
        BRGame.RestartBRGame();
    }
}

//called when Delete clicked in settings screen
reliable server function DeleteSettings(string map)
{
    local UTBRSettings settings;
    
    if (! AllowAdminAccess())
    {
        return;
    }
       
    settings = new (none, GetSettingsName(map)) class'BombingRun.UTBRSettings';   
    settings.ClearConfig();
    
    if (BRGame != none)
    {
        BRGame.ApplySettings();
    }
}

//called when Save clicked in settings screen
reliable server function SaveSettings(string map, string serializedData, string mutators)
{
    local UTBRSettings settings;
    
    if (! AllowAdminAccess())
    {
        return;
    }
       
    settings = new (none, GetSettingsName(map)) class'BombingRun.UTBRSettings';
    Settings.DeSerialize(serializedData, mutators);
    Settings.SaveConfig();

    if (BRGame != none)
    {       
        BRGame.ApplySettings();
    }
}

reliable Server function SetDebug(string debugStr)
{
    if (! AllowAdminAccess())
    {  
        return;
    }

    UTBRGame(WorldInfo.Game).SendDebugMessage(PlayerReplicationInfo.GetPlayerAlias() $ " invoked SetDebug " $ debugStr);
    
    UTBRGame(WorldInfo.Game).SetDebugStr(debugStr);
}

function InitSpecialNavigation(optional bool fullInit, optional bool midInit)
{
    GoalJumping = false;
    GoalJumpingWithBall = false;
    TryJumpShot = false;
    TurningForGoalJumpShot = false;    
    JumpStartedByPlayerManager = false;
    SpecialNavigationPathTarget = none;
    
    if (! midInit)
    {
        //midInit means we are initing while special nav is in progress and
        //rerouting it on a new path
        WaitingForMultiJumpApex = false;
        SpecialNavigating = false;            
    }      
    
    if (fullInit)
    {
        //done to totally init special navigation.
        //if false then only inits needed for continuing navigation are done.
         
        FindingRandomTarget = false;          
        Destination = vect(0,0,0);
        LastJumpStartLocation = vect(0,0,0);
        TurnOffBlocking();
        JumpingOverBlockage = false;
        JumpOverBlockage = false;                
    }      
}

function TurnOffBlocking()
{
    BackingUp = false;
    ZStrafing = false;
    Blocked = false;
    ZBlocked = false;
}

function SetPathSubtarget(Actor target)
{
    local RouteInfoStruct ri;
    
    if (PathSubTarget != target)
    {
        //since a new path is being made, flush route history to avoid
        //accidentally detecting a loop from old path
        RouteHistory.Length = 0;
        
        PathSubTarget = target;

        BRNodeRouteInfo = ri;
    }
}

//set the path destination of bot
function SetPathTarget(Actor target, optional bool directPathTo)
{
    if (PathTarget != target)
    {
        RouteHistory.Length = 0;
        MoveTarget = none;
        TriedForcedSpecialNavigationToPathTarget = false;
        SetPathSubTarget(none);
      
        if ((target != none) && (target == Bot.DefensivePosition))
        {
            TriedDefensivePosition = true;
            CanUseAssaultPaths = true;            
        }  
    }
    
    PathTarget = target;
    
    if (directPathTo)
    {
        SetMoveTarget(target);     
    } 
}

//return true if bot is trying to reach formation center rather than
//on a side task such as getting a weapon.
function bool IsSeekingFormationCenter()
{
    return IsFormationCenter(PathTarget) || (PathTarget == Bot.DefensivePosition);
}

//return true if target is formation center
function bool IsFormationCenter(Actor target)
{
    return (FormationCenter == target) || (FormationCenterTarget == target);
}

//find a good random place to strafe to
function FindRandomDestination()
{
   //Bot.FindRandomDest() doesn't always work and just returns none.
   //so we must have our own routine.
   //this routine works a lot better and faster than randomly selecting a path node.
 
   local rotator r;

   r.yaw = 8192 * rand(8);
   RandomDestination = Bot.Pawn.Location + (vector(r) * 4000);   
}

//set what bot is turned/aiming at
function SetFocus(Actor target)
{
    if (target == none)
    {
        return;
    }

    Bot.Focus = target;

    //bot firing code needs both focus and focalpoint set    
    if (target == Bot.Enemy)
    {
        Bot.FocalPoint = target.Location;
    }
    else if ((UTBRBall(target) != none) && (AttackBallBoosting || DefenseBallBoosting))
    {
        Bot.FocalPoint = target.Location;
    }     
    else
    {
        //otherwise focus just overrides focalpoint so it's not needed
        Bot.FocalPoint = vect(0,0,0);
    }
    
    if (UTBRGoal(target) != none)
    {
        //aim at center of goal so bot can shoot ball at goal when needed.
        SetFocalPoint(target.Location);
    }
}

//set location bot is aiming at
function SetFocalPoint(Vector target)
{                 
    Bot.Focus = none; //focus will override focalpoint unless we set it to none 
    Bot.FocalPoint = target;  
}

//used for accuracy when firing ball launcher at something. FocalPoint should
//already be set and weapon's aim should already be close.
function SetAim(Vector target)
{
    Bot.Pawn.SetViewRotation(rotator(target - Bot.Pawn.GetPawnViewLocation()));
}

//set location bot will aim at and shoot ball.
//uses AimTimer to make bot position ball launcher in real time, during
//which time players can see the bot adjusting it's aim and try to frag the bot.
//just overriding the bot view rotation is too fast and not very realistic.
function SetupBallShot(Vector target)
{   
    AimFocalPoint = target;
    SetFocalPoint(target);
    AimTime = WorldInfo.TimeSeconds;
    SpecialAiming = true;
    SetBotTacticTimer();
}

//check and see if bot is using a jump pad.
//retask is not done so we must manually set the movetarget.
function bool CheckJumpPad()
{   
    if (Bot.Pawn.Physics == PHYS_Falling)
    {
        if ((UTJumpPad(MoveTarget) != none) && (Bot.MoveTarget == UTJumpPad(MoveTarget).JumpTarget))
        {
            ShowPath("Using jump pad " $ MoveTarget $ ". New MoveTarget=" $ Bot.MoveTarget);
                        
            SetAsReached(MoveTarget);
            
            SetMoveTarget(Bot.MoveTarget);
            
            if (Bot.Enemy == none)
            {
                SetFocus(MoveTarget);
            }
                      
            return true;
        }
    }
    
    return false;
}

function SetMoveTarget(Actor target)
{
    if (MoveTarget != target)
    {
        LastMoveTarget = MoveTarget;
    }
            
    if ((MoveTarget != target) || (RouteHistory.Length == 0))
    {
        ReachedTarget(MoveTarget);  //set if target was reached or not
            
        AddToRouteHistory(target); 

        MoveTargetTime = WorldInfo.TimeSeconds;
    }
 
    MoveTarget = target;
    SetUTBotMoveTarget();  
    
    DistFromMoveTarget = UTBRGame(WorldInfo.Game).DistanceBetween(Bot.pawn, MoveTarget);
}

function bool IsFailedNode(Actor target, optional int index)
{
    return 
    (RouteHistory.Length > index) &&
    (RouteHistory[index].target == target) && 
    (! RouteHistory[index].Reached);
}

function bool IsReachedNode(Actor target, optional int index)
{
    return 
    (RouteHistory.Length > index) &&
    (RouteHistory[index].target == target) && 
    (RouteHistory[index].Reached);
}

//return true if bot has already visited target along current path
function bool HasReached(Actor target)
{
   local int i;
     
   for (i = 0; i < RouteHistory.Length; i++)
   {  
       if (IsReachedNode(target, i))
       {
           return true;
       }
   }
   
   return false;
}

//return true if should not be allow to revisit target (loop detection)
function bool DisallowRevisit(Actor target)
{
   local int i;

   if (Bot.Pawn.Physics == PHYS_FALLING)
   {
       //sometimes when reach a node as falling ut will go back to it so wait 
       //till stop falling
       return false;
   }
   
   for (i = 0; i < RouteHistory.Length; i++)
   {  
       if (IsReachedNode(target, i))
       {
           if (RouteHistory[i].AllowRevisit)
           {
               ShowPath("Allowing revisit to node " $ target);
               RouteHistory[i].AllowRevisit = false;
           }
           else
           {
               return true;
           }
       }
   }
   
   return false;
}

//return true if target is in history 
function bool IsInRouteHistory(Actor target)
{
   local int i;
     
   for (i = 0; i < RouteHistory.Length; i++)
   {  
       if (RouteHistory[i].target == target)
       {
           return true;
       }
   }
   
   return false;
}

//set all nodes to allow revist because detouring to target
function SetAllowRevist(Actor target)
{
   local int i;
     
   for (i = 0; i < RouteHistory.Length; i++)
   {  
       if (RouteHistory[i].target != target)
       {
           RouteHistory[i].AllowRevisit = true;
       }
   }
}

function AddToRouteHistory(Actor actorToAdd)
{
    local int routeHistoryLength;
    
    //only write node once while attempting to reach
    if ((RouteHistory.Length > 0) && (actorToAdd == RouteHistory[0].Target))
    {
        return;
    }
    
    routeHistoryLength = 3;
    RouteHistory.Insert(0, 1);
    if (RouteHistory.Length > routeHistoryLength)
    {
        RouteHistory.length = routeHistoryLength;
    }

    RouteHistory[0].Target = actorToAdd;
    RouteHistory[0].Reached = false;     
}

function SetAsReached(Actor actorReached)
{   
    if ((RouteHistory.Length > 0) && (RouteHistory[0].Target == actorReached))
    {
        RouteHistory[0].Reached = true;
        
        if (Bot.Pawn.Physics == PHYS_Falling)
        {
            //sometimes bot will go back to movetarget if reached when falling.
            //even when bot is back on ground it might go back once.
            //allow one revisit once back on ground.
            RouteHistory[0].AllowRevisit = true;
        }
    }
}

//return true if bot has has reached target.
//can not use ut's ReachedDestination during our special navigation, 
//as it is unreliable for that purpose. Seems to work best only with
//ut3 movement functions.
function bool ReachedTarget(Actor target)
{
    local float cRadius, cHeight, cRadius2, cHeight2;
    local bool reached;

    if (target == none)
    {
        return false;
    }
    
    //not so good to do this. once bot starts falling/moving away from target, we
    //should no longer consider it reached.
    //consider a bot who reaches target mid air and then starts falling away, as
    //it drifts out of reach then should no longer consider as reached.
    //not to be confused with RouteHistory which keeps track of whether or not bot
    //had reached target at all. 
    //reached = IsReachedNode(target);
   
   
    //still use the ut3 function for optimization and failsafe purposes.
    //between the ut3 function and br logic we ought to be able to reliably detect 
    //when reached target.
    reached = Bot.Pawn.ReachedDestination(target);
    
    if (! reached)
    {
        //br detection logic will be to detect when collision cylinders intersect horizontally
        //or vertically. very important especially if in vehicle and target is against a 
        //wall. bot may not be able to actually get right on the target but the collision
        //boundaries should at least intersect.
        //sadly, TouchingActors interation does not work even though bot
        //may be actually touching it.
    
        //typical body radius is 21, height 44 (height is half the length head to toe)
        Bot.Pawn.GetBoundingCylinder(cRadius, cHeight);
        
        target.GetBoundingCylinder(cRadius2, cHeight2);
        if (UTBRGoal(target) != none)
        {
            //goal won't have an accurate collision cylinder so just
            //use a small center area 
            cRadius2 = 10;
            cHeight2 = 25;
        }    
        
        cRadius += cRadius2;
        cHeight += cHeight2;

        //make the collision boundaries overlap a bit so bot is clearly touching        
        if (cRadius > 15)
        {
            cRadius -= 5;
        }
        
        if (cheight > 15)
        {
            cHeight -= 5;
        }
                         
        if ((UTBRGame(WorldInfo.Game).HorizontalDistanceBetween(target, Bot.Pawn) <= cRadius) &&
            (Abs(UTBRGame(WorldInfo.Game).VerticalDistanceBetween(target, Bot.Pawn)) <= cHeight))
        {
            reached = true;
        }
    }
    
    if (reached && (MoveTarget == target))
    {
        SetAsReached(target);
    }
    
    return reached;
}

//called by UTBRSquadAI
function Retask()
{
    /*
    many reasons for managing retask ourselves.
    
    we want dynamic retasking wherein retasking can be called anytime
    to refresh bot ai without threat of disturbing bot's current path.
    bot's should be able to reevaluate their objectives at any time and 
    make any needed adjustements without destroying what they are currently
    doing.
    
    limit how often can retask. ut allows retasking to happen too often
    and quite a few can happen every second which can be expensive.
    they do have some limits in their own retask function but it decreases
    bot response time. for br bot response time is always
    immediate unless retasking is being called too many times per second.
    
    in ut3 retask they don't retask if bot is falling but in
    BR we allow this. we want bot to reroute mid air.

    ut3 gives bot too much time to continue current
    move so for br we take this time down. with stock ut logic
    bots can just blindly plow into jumppad when they ought to 
    be changing their route immediately.     

    in ut retask there's all this complicated stuff manipulating bot's
    movement state. we don't do this for br. does not seem to be needed
    and not sure why they do it. it's also convoluted and hard to manage.
    anyway, for br we don't want bot's movement to be disturbed by a retask.   
    */

    Retasking = true;
        
    //must delegate off to a timer to avoid infinite recursion
    //not sure why ut3 delegates the retask to the physics tick
    //but here we just put off to normal timer.
    if (! IsTimerActive('Retask2'))
    {
        SetTimer(0.01, false, 'Retask2');
    }
}

//called by Retask        
function Retask2()
{
    if ((Bot != none) && (Bot.Pawn != none) && (Bot.Squad != none))
    {
        if (Bot.Pawn.Physics == PHYS_Falling)
        {
            //ut ignores retask when falling so must call direct,
            //sacrificing some functionality we won't get by not calling WhatToDoNext.
            Bot.Squad.AssignSquadResponsibility(Bot);
        }
        else
        {
            Bot.WhatToDoNext();
        }
    }
}

/*
called by UTBRSquadAI.
must use UTBRPlayerManager to manage this because ut can call it way too many
times per second(lots and lots). can't control this via Retask because it's also
called directly without using Retask. we control here how many times per second it's called. 
*/
function AssignSquadResponsibility()
{
    local float tickTime;
    
    if (! IsTimerActive('AssignSquadResponsibility2'))
    {
        tickTime = 0.01;
        if ((WorldInfo.TimeSeconds - LastRetaskTime) < 0.25)
        {
            tickTime = 0.25;
        }
        LastRetaskTime = WorldInfo.TimeSeconds;
            
        SetTimer(tickTime, false, 'AssignSquadResponsibility2');
    }
}

function AssignSquadResponsibility2()
{
    if (Retasking)
    {
        Retasking = false;
        
        ShowPath("Retask");
    }
    else
    {
        //when UT needs to refresh the path it calls AssignSquadResponsibility but does not call Retask.
        //path refresh happens when bot reaches movetarget and many time will also happen before bot
        //reaches movetarget for whatever reason.
        //when this happens, it's important to allow a fresh path to be calculated, because for whatever
        //reason ut is saying it needs fresh path calls to be made or else problems will occur.
        //even if the MoveTarget result would be exactly same as before, if a fresh FindPathToward() is not
        //called, the bot can totally miss the movetarget! bot would reach movetarget but does not detect 
        //it and moves beyond it, and then later on when AssignSquadResponsibility is called again, bot must
        //go backwards to the missed target. seems like a ut oddity or bug.
         
        if (! SpecialNavigating)
        {
            ForceRepath = true;
        }
    }
    
    UTBRSquadAI(Bot.Squad).AssignSquadResponsibilityCallBack(Bot);
}  

//go into spectate mode. only for admins and instant action.
//for other online players it could be used to spy on enemy so we don't allow it. 
reliable Server function Spectate()
{
    if (! AllowAdminAccess())
    { 
        return;
    }
           
    if (PlayerReplicationInfo.bIsSpectator)
    {
        UTBRGame(WorldInfo.Game).SendDebugMessage(PlayerReplicationInfo.GetPlayerAlias() $ " has left Spectate mode");
        
        PlayerReplicationInfo.bIsSpectator = false;
        PlayerReplicationInfo.bOnlySpectator = false;        
        Controller.GotoState('PlayerWaiting');
        Controller.ServerRestartPlayer();  
    }
    else
    {
        UTBRGame(WorldInfo.Game).SendDebugMessage(PlayerReplicationInfo.GetPlayerAlias() $ " has gone into Spectate mode");
        
        //keeping the pawn in online play could perhaps be exploited, and also can cause errors,
        //but allow it in instant action so you can spec your own pawn
        if ((Controller.Pawn != none) && (WorldInfo.Netmode!=NM_Standalone))
        {
            Controller.Pawn.Suicide();
        }
        
        Controller.GotoState('Spectating');
        PlayerReplicationInfo.bIsSpectator = true;
        PlayerReplicationInfo.bOnlySpectator = true;
    }            
}

simulated function PostBeginPlay()
{   
    super.PostBeginPlay();
    
    BRGame = UTBRGame(WorldInfo.Game);
        
    if (BRGame != none)
    {      
        BRGame.PlayerManagerArray[UTBRGame(WorldInfo.Game).PlayerManagerArray.Length] = self;
    }

    SetTimer(PMTimerInterval, true, 'PMTimer');    
}


simulated event ReplicatedEvent(name VarName)
{   
    if (VarName == 'PlayerReplicationInfo')
    {
        //sometimes a ut quirk causes pri to change at server, so on client we must turn timer back on
        //so it can check if this is the local player. not doing this will cause the SpectatorTimer to not work.
        LocalPlayerController = none;
        SetTimer(PMTimerInterval, true, 'PMTimer');    
    }
}

//run on client for local player in net game, and on server
simulated function PMTimer()
{
    local PlayerController PC;
    local Weapon w;

    BRGame = UTBRGame(WorldInfo.Game);
       
    if (BRGame == none)
    {
        return;
    }
    
    if (Bot != none)
    {
        Squad = UTBRSquadAI(Bot.Squad);
    }
    
    if ((Controller != none) && (PlayerReplicationInfo != Controller.PlayerReplicationInfo)) 
    {
        //sometimes pri can change in weird situations      
        PlayerReplicationInfo = Controller.PlayerReplicationInfo;
    }
      
    if (PlayerReplicationInfo == none)
    {
       return;
    }
        
    if (
        (Role < ROLE_Authority) && (Controller != none) && 
        (Controller.Pawn != none) && UTBRGame(WorldInfo.Game).GameEnded &&
        (UTWeapon(Controller.Pawn.Weapon) != none)
       )
    {
        //must do this or if winner was jumping they see the first person arms mesh in winner pic
  		UTWeapon(Controller.Pawn.Weapon).ChangeVisibility(false);  
    }
               
    if ((Role == ROLE_Authority) && (Controller != none) && (Controller.Pawn != none))
    {
   
        //used for display of player on minimap
        PawnLocation = Controller.Pawn.Location;
        PawnRotation = Controller.Pawn.Rotation;
        
        PawnVisibleOnMiniMap = 
            (! Controller.Pawn.bHidden) && (Controller.Pawn.Health >0) && 
            (! Controller.Pawn.IsInvisible()) && 
            ((UTVehicle(Controller.Pawn) == none) || (UTVehicle_Hoverboard(Controller.Pawn) != none));
            
        //pri kills don't replicate, so we must do so here
        Kills = PlayerReplicationInfo.Kills;
        
        //if we don't replicate this then player can't do the extra jumps
        if (UTPawn(Controller.Pawn) != none)
        {
            MaxMultiJump = UTPawn(Controller.Pawn).MaxMultiJump;
        }
        
        if (UTVehicle(Controller.Pawn) != none)
        {
            LastVehicle = UTVehicle(Controller.Pawn); 
        }        
    }
           
   
    //fix for some ut3 quirk. sometimes on client pc after map transition the playercontroller state is still
    //roundended instead of playerwaiting, which causes player to not be able to ready up
    if ((Role < ROLE_Authority) && (LocalPlayerController != none) && 
        (UTBRGame(WorldInfo.Game) != none) && (UTBRGame(WorldInfo.Game).IsPendingMatch) &&
        (LocalPlayerController.GetStateName() == 'RoundEnded'))
    {
        LocalPlayerController.GotoState('PlayerWaiting');
    }
        
    if ((Role < ROLE_Authority) && (LocalPlayerController != none) && (UTPawn(LocalPlayerController.Pawn) != none))
    {
         if (UTPawn(LocalPlayerController.Pawn).MaxMultiJump != MaxMultiJump)
         {     
             UTPawn(LocalPlayerController.Pawn).MaxMultiJump = MaxMultiJump;
             UTPawn(LocalPlayerController.Pawn).MultiJumpRemaining = MaxMultiJump;
         }       
    }
       
    if ((LocalPlayerController == none) && (WorldInfo.Netmode == NM_Standalone))
    {         
        foreach WorldInfo.AllControllers(class'PlayerController', PC)
        {
      	    LocalPlayerController = PC;
                      
    	    if (PC.PlayerReplicationInfo == PlayerReplicationInfo)
    	    {
    	        Controller = PC;
    	        UTBRGame(WorldInfo.Game).LocalPlayerManager = self;
    	    }        

            break;
        }     
    }
        
    if ((LocalPlayerController == none) && (Role < ROLE_Authority))
    {
        if (UTBRGame(WorldInfo.Game) == none)
        {
           return;
        }
            
        foreach WorldInfo.AllControllers(class'PlayerController', PC)
        {
        	if (PC.IsLocalPlayerController())
        	{
        	    LocalPlayerController = PC;
        	    
        	    if (PC.PlayerReplicationInfo == PlayerReplicationInfo)
        	    {
        	        Controller = PC;
        	        UTBRGame(WorldInfo.Game).LocalPlayerManager = self;
        	    }
        	    else
        	    {          	    
        	        //turn off for all but local player
        	        SetTimer(0.0, false, 'PMTimer');
                    return;        	    
                }
        	}
        }
        
        if (LocalPlayerController != none)
        {
            GetServerGameSettings();       
        }
    }       
   
    if ((TheBall != none) && (TheBall.IsHeldBy(Controller.Pawn)))
    {
        TheBall.SetOwnerNoSee(Controller.Pawn.IsFirstPerson() && (Vehicle(Controller.Pawn) == none));
    }  

    if (Role == ROLE_Authority)
    {
        BotAITimer(true);     
        
        BotAiTime = BotAiTime + PMTimerInterval;
        
        if (BotAiTime >= 1.0)
        {
            BotAITimer(false);
            BotAiTime = 0;
        }       
    }

    if (UTBRGame(WorldInfo.Game).Test_GiveBall && (WorldInfo.Netmode == NM_Standalone))
    {
        //testing only. ball is given to player holding impact hammer.
        foreach WorldInfo.AllControllers(class'PlayerController', PC)
        {
            if ((PC.Pawn != none) && (UTWeap_ImpactHammer(PC.Pawn.Weapon) != none))
            {
                if (UTBRGame(WorldInfo.Game).TheBall.IsHeld() && (UTBRGame(WorldInfo.Game).TheBall.HolderController != PC))
                {
                    UTBRGame(WorldInfo.Game).TheBall.SendHome(none);
                }
                
                ForEach PC.Pawn.InvManager.InventoryActors( class'Weapon', w)
                {
                    if ((UTWeap_ImpactHammer(w) == none) &&
                        (UTWeap_Redeemer(w) == none))
                    {
                        PC.Pawn.SetActiveWeapon(w);
                        break;
                    }
                }                
             
                if (UTBRGame(WorldInfo.Game).TheBall.HolderController != PC)
                {   
                    UTBRGame(WorldInfo.Game).TheBall.SetHolder(PC);
                }
                
                BallLauncher.PendingWeapon = w;                
            }
            break;
        }
    }     
            
    if ((PlayerReplicationInfo != none) && (PlayerReplicationInfo.bOnlySpectator) && 
        (! IsTimerActive('SpectatorTimer')))
    {   
        SetTimer(0.01, True, 'SpectatorTimer');
    }
}

//tick for special bot ai. this is called outside of the regular ut ai tick because 
//we can't control that tick, and it generally only gets called when bot needs to retask,
//such as when they reach a travel destination. we need a more fine grained timer that that.
function BotAITimer(bool DoFineGrainedChecks)
{  
    if (Bot == none)
    {
        return;
    }
    
    BRBotAI(DoFineGrainedChecks);
}

function SetBotTacticTimer()
{
   if (! IsTimerActive('BotTacticTimer'))
   {
      SetTimer(0.05, true, 'BotTacticTimer');
   }
}

function ClearBotTacticTimer()
{
   ClearTimer('BotTacticTimer');
}

/*
perform ai to be performed once per second.
check and see if bot will shoot/pass ball, do special jumps, etc.

CheckSquadObjectives is called in a tick which only gets evaluated at times we can't control,
such as when bot reaches a destination. sometimes the ut ai tick is fired many times a second, and
sometimes several seconds can be between ticks. 
so, we use our own ai tick which we can control to give bot special tactics which are needed 
at any given moment. yet we can also keep the tick down to a reasonable time so that the logic
does get evaluated too often for perfomanace reasons, and to keep bots from doing certain moves
too many times in a row.
also, this can't be called during squadai tick anyway because weapon firing is disabled there.
*/
function BRBotAI(bool FineGrainedChecks)
{       
    if ((Bot == none) || (Bot.Pawn == none))
    {
        return;
    }  
    
    if (WorldInfo.Game.bGameEnded)
    {
        return;
    }

    SetVars();
    
    if (FineGrainedChecks)
    {
        DoFineGrainedChecks();
        return;
    }
        
    OldHealth = CurrentHealth;
    CurrentHealth = Bot.Pawn.Health;
    GoodFloorDist = CheckForFloorFacing();

    if (CheckGeneralSoak())
    {
        return;
    }   
       
    CheckIfReachedFallback();
    CheckBrNodeRadius();  
    
    CheckObjectiveJump();

    if (abs(Bot.Pawn.Rotation.Yaw - LastYaw) >= 1000)
    {
        RotatingBigVehicleTime = WorldInfo.TimeSeconds;
    }  
    
    if (vsize(Bot.Pawn.Velocity) >= 50)
    {
        RotatingBigVehicleTime = 0;
    }
       
    LastYaw = Bot.Pawn.Rotation.Yaw;  

    if (GetBall(Bot.Pawn) == none)
    {
        return;
    }  
    //rest of functions for ball carriers only
        
    if ((UTPawn(Bot.Focus) != none) && (MoveTarget != none) && (! SpecialFocus) &&
        (! CanUseWeaponsWithBall()))        
    {
       //stop runner from looking back.
       //normally in ut this would be good because bot would have weapons and be shooting
       //backwards, but not so good for ball runner.
       SetFocus(MoveTarget);
    }
    
    /*
    this seems to be ok now. early on had a prob with bots on hoverboards in 
    HerosOfAnubis not scoring but they are getting off board on their own now. 
    now that we allow to carry orb with vehicles bot should be allow to score with vehcile if they can.
    if ((HorizontalDistanceFromEnemyGoal <= 1000) && (Vehicle(Bot.Pawn) != none))
    {
       
        Bot.LeaveVehicle(true);
    }
    */

    if (CheckBallReset()) return;                        
    if (CheckGoalShot()) return;
    if (CheckLobPass()) return;           
    if (CheckPlayerPass()) return;
    if (CheckForwardPass()) return;  
}

//these checks are done many times per second.
//we don't want to do too much here as it would be too expensive,
//but some things are needed here.
function DoFineGrainedChecks()
{       
    if (
        (! SpecialNavigating) &&
        (Bot.Pawn.Physics == PHYS_Falling) &&
        (MoveTarget != none)
       )
    {
        /*
        use BR jump navigation to navigate bot in air. 
        much better than ut's.
        this special navigation will take advantage of multijumps and continuously navigate
        bot towards their target.
        note Bot.MoveTarget is not used. Sometimes ut can try to sneak in a new movetarget suddenly
        which didn't come from br pathing. we need to control the MoveTarget since
        there's so many bugs in ut and we need to keep the bot on route.
        it's unbelievable how much better pathing becomes when we force the movetarget to
        not change until it's reached or deemed unreachable. before we started doing this,
        bots were always trying to get into looping conditions, falling off bridges, etc etc.
        
        because playerManager.ContinuePath() keeps bot on route and heading to movetarget, 
        we must keep movetarget unchanged(or be very certain the change is correct) 
        or the bot can get in loops. integrity checks on the route
        are not done until ContinuePath() lets a new movetarget be selected.
        */

        CheckJumpPad();
                       
        StartSpecialNavigation(MoveTarget);     
    } 
    
    CheckKillZReset();
    
    if (CheckFormationCenterChanged())
    {
        Retask();
    }
    else if (CheckFormationCenterMoved())
    {
        Retask();
    }    
       
    if ((UTBRGame(WorldInfo.Game).TestBotFocus != vect(0,0,0)) && (! SpecialAiming))
    {
        SetFocalPoint(UTBRGame(WorldInfo.Game).TestBotFocus);
    }
       
    if ((UTVehicle_Goliath(Bot.Pawn) != none) && (! SpecialNavigating) &&
        (MoveTarget != none) &&
        IsFailedNode(MoveTarget) && ReachedTarget(MoveTarget))
    {
        //sometimes silly ut doesn't detect/report target reached when in tank.
        
        ShowPath("Detected tank reached " $ MoveTarget);
        
        /*
         for now just want to know when tank reaches something that ut doesn't report.
         retasking here seems to make the tank have tendency to run into walls.
                   
         Retask();
        */
    } 
}

//check if bot will try to jump to objective
function bool CheckObjectiveJump()
{
    local int i;
    local float floorDist;
    local bool ret;
    local Vector X,Y,Z;
    
   
    if ((! CanInitiateJump()) || TurningForGoalJumpShot)
    {
        return false;
    }

    if (DoingBallReset)
    {           
        //situation could be that bot is facing killzone but backing up to a jumppad
        //it needs to use to reach killzone. in cases like that where bot is not facing
        //it's movetarget, we should not jump at killzone as that would take bot away
        //from movetarget
        
        if ((rand(2) == 0) &&
           (MoveTarget != none) &&
           FacingGeneralDirection(MoveTarget.Location) &&                                  
           (DistanceBetween(Bot.pawn, Squad.FriendlyGoalInfo.BallResetTarget) > 500) && (GetBall(Bot.Pawn) != none))
        {
            if (TryJump(Squad.FriendlyGoalInfo.BallResetTarget)) return true;
        }
    
        return false;
    }       
    
    i = 4;
    
    if ((Bot.Skill >= 4) && (Bot.Skill <= 6))
    {
        i = 3;
    }
    
    if (Bot.Skill > 6)
    {
        i = 2;
    }    

    floorDist = 1000;
    if (IsLowGravity()) 
    {
        floorDist = 1500;
    }

    if ((rand(i) == 0) && (! ShootingGoal) && 
        (GoodFloorDist >= floorDist) &&
        (HorizontalDistanceFromFormationCenter > floorDist) &&
        (! Bot.Pawn.NeedToTurn(FormationCenterLocation)) &&
        IsFormationCenter(PathTarget)
       )
    {           
        if (
            ((UTBRBall(FormationCenter) != none) && (! UTBRBall(FormationCenter).IsHeldByTeamOf(Bot.Pawn))) ||
            (FormationCenter == Squad.EnemyGoal)
           )
        {
            GetAxes(Bot.Pawn.Rotation,X,Y,Z);

            //don't dodge into blockage            
            if (BRGame.BlockedDist(Bot.Pawn, X, floorDist, Bot.Pawn.Location, vect(25,25,25)) == 0)
            {
            
                PerformForwardDodge(FormationCenterTarget);
                return true;
            }           
        }
    }

    i = 4;

    if (HorizontalDistanceFromFormationCenter > 1000)
    {
       i--;
       TriedPath = false;       
    } 
        
    if (TriedPath)
    {
       //if bot has already tried getting to objective then make it more likely they will jump
       i--;
    }  
    
    if ((UTBRBall(FormationCenter) != none) && 
        UTBRBall(FormationCenter).IsInMotion() && 
        (VerticalDistanceFromFormationCenter > 200))
    {
        //ball or holder is in flight. want bot to be aggressive in pursuing ball but don't want them too jumpy.
        i = 4;
    }

    if (UTBRGame(WorldInfo.Game).Test_JumpyBots)
    {
        i = 0;
        VerticalDistanceFromFormationCenter = 300;
    }
    
    if (
        ((WorldInfo.TimeSeconds - LastJumpTime) > 4) &&    
        (ShouldPickupBall() || (FormationCenter == Squad.EnemyGoal)) &&
        (rand(i) == 0) &&          
        CanSeeObjective() && 
        (VerticalDistanceFromFormationCenter > 200) &&
        AllowJumpToFormationCenter() &&
        (! Bot.Pawn.NeedToTurn(FormationCenterLocation)) &&
        (! GettingSpecialItem()) &&                
        (TakingDefensivePathTime == 0)
       )
    {                        
        ShowPath("Performing random jump. i=" $ i);
             
        ret = TrySpecialJump(FormationCenter);
                
        if (ret)
        {
            SuppressLoopChecks = true;      
        }        
        
        return ret;
    }
    
    return false;  
}

//check and see if bot will try to reset ball
function bool CheckBallReset()
{    
    //see if bot near home goal will reset ball by shooting it into nearby killzone    
    if ((! DoingBallReset) && CanResetBall &&
        (HorizontalDistanceFromFriendlyGoal <= BallResetDist) &&
        CheckTrace(CAN_SEE, Squad.FriendlyGoal) && 
        (Squad.FriendlyGoalInfo.KillZone != none) &&      
        OkToResetBall())
    {
        ShowPath("Resetting ball");
            
        DoingBallReset = true;
        ForceDefensivePosition = false;
        Bot.DefensivePosition = none;        
        ResetBallTime = WorldInfo.TimeSeconds;
        SetVars();        
        Squad.SeekObjective(self);        
        SetBotTacticTimer();
        Retask();
    
        return true;
    }    
    
    return false;
}

//check and see if bot will try to shoot ball into enemy goal
function bool CheckGoalShot()
{
    local bool TryShoot;
 
    //check to see if bot will shoot ball into goal.
    //no need for shootspots with this logic.
    TryShoot = false;
    
    if (IsUnderStress())
    {
        //make bot shoot if they are under heavy attack
        TryShoot = true;
    }

    if (
         (rand(3) == 0) &&
         (Squad.DistanceFromEnemyGoal(Bot.Pawn) > 500) &&
         IsThreatened()
       )
    {
        //bot may try a shot, driven by a 33% chance.
        //they must be threatened, so they must see enemy and they must have taken some damage or 
        //match must be instagib(so death is frequent).
        //also, don't bother if they are so close to goal they might be able to just jump in.
        //IsUnderStress test above would force the shot if they were in bad health so here they have
        //higher health and have good chance of making it.
        TryShoot = true;
    }
                    
    if (! CheckTrace(CAN_SEE, Squad.EnemyGoal))
    {
        //must must be able to see goal to try a shot
        TryShoot = false;
    }
     
    if (ShootingGoal)
    {
        //no need to try if already trying
        TryShoot = false;       
    }
    
    if (bool(UTBRGame(WorldInfo.Game).Settings.DisableShootingScore))
    {
        TryShoot = false;
    }
                   
    if (TryShoot)
    {   
        //bot will try to shoot ball into enemy goal for score
        TryShootingGoal(Squad.EnemyGoal);
        return true;
    }    
    
    return false;
}

//return true if bot should not pass ball
function bool AvoidPassing()
{
    if (SpecialAiming)
    {
        return true;
    }
    
    return
        (UTVehicle(Bot.Pawn) != none) &&
        (UTVehicle_Hoverboard(Bot.Pawn) == none);
} 

//check and see if bot near home goal will lob ball far away
function bool CheckLobPass()
{
    local Vector target, dir;

    if (AvoidPassing())
    {
        return false;
    }
    
    //check and see if bot near home goal will lob ball far away
    if (
        (! Squad.FriendlyGoal.DisableLobbingBallAway) &&
        (HorizontalDistanceFromFriendlyGoal < LobPassDistance) &&
        FacingGeneralDirection(Squad.EnemyGoal.Location)
        )
    {
        //get direction to point gun   
        dir = GetLobDirection();
                          
        if (dir == vect(0,0,0))
        {
            return false;
        }
               
        target = Bot.Pawn.GetPawnViewLocation() + (dir * 1000);
                   
        if (PrimeLauncher())
        {
            return false;
        }
        
        ShowPath("Performing lob pass");        
       
        SetupBallShot(target);
           
        return true;
    }    
          
    return false;
}
  
//check and see if bot will pass ball forward.
function bool CheckForwardPass()
{
    local bool TryShoot;
    local Vector v;
    local float dist, floorDist;
    local UTOnslaughtNodeObjective node;
    
    if (AvoidPassing() || BRGame.Test_NoShoot)
    {
        return false;
    }
      
    //only release ball if they are pointing towards base, as shooting backwards is not good.   
    
    TryShoot = false;
    if ((rand(3) == 0) && ((WorldInfo.TimeSeconds - Bot.LastSeenTime) <= 4))
    {
        TryShoot = true;
    }
    
    if (IsThreatened() && (rand(2) == 0))
    {
        TryShoot = true;
    }
    
    if (IsUnderStress())
    {
        TryShoot = true;
    }
            
    if (HorizontalDistanceFromFriendlyGoal < LobPassDistance)
    {
        //will wait to setup a lob pass
        TryShoot = false;
    }
    
    if (
        (UTHoldSpot(Bot.DefensePoint) != None) ||
        GoalJumping ||
        ShootingGoal      
       )        
    {
        TryShoot = false;
    }

    dist = 1000;
    if (IsLowGravity())
    {
        dist = dist + 1000;
    }
    
    if (Squad.EnemyGoalInfo.KillZone != none)
    {
        dist = dist + 500;
    }

    floorDist = 1500;
    if (IsLowGravity())
    {
       floorDist = 2000;
    }
                          
    if (TryShoot)
    { 
        TryShoot =
        FacingGeneralDirection(Squad.EnemyGoal.Location) &&
        (HorizontalDistanceFromEnemyGoal > dist) &&             
        CheckTrace(CLEAR_SHOT_PATH) && 
        (GoodFloorDist >= floorDist);       
    }
    
    //decide if bot should release ball before attending to node
    node = UTOnslaughtNodeObjective(FormationCenter);
    if (
        (node != none) && 
        (GetBall(Bot.Pawn) != none) &&
        (DistanceBetween(node, Bot.Pawn) <= 1000)      
       )
    {
        if (node.IsNeutral())
        {
            //bot can just touch node to activate
            TryShoot = true;
        } 
        else if (
                 (node.GetTeamNum() == Bot.GetTeamNum()) && 
                 HasHealingWeapon(node) &&
                 (! bool(UTBRGame(WorldInfo.Game).Settings.AllowBallCarrierWeapons)) 
                )
        {
            //but has a healing weapon. use it instead of destroying ball.
            TryShoot = true;
        } else if (
                   (! NeedWeapon()) &&
                   ((UTVehicle(Bot.Pawn) == none) || (UTVehicle_Hoverboard(Bot.Pawn) != none)) &&
                   (! bool(UTBRGame(WorldInfo.Game).Settings.AllowBallCarrierWeapons))                   
                  )
        {
            //bot can destroy node without destroying ball
            TryShoot = true;
        }        
    }
       
    if (TryShoot)
    {
        if (PrimeLauncher())
        {
           return false;
        }
        
        ShowPath("Performing forward pass");

        //point gun down for shot     
        v = Bot.Pawn.Location + (vector(Bot.Pawn.Rotation) * 1000);
        v.Z = v.Z - Bot.Pawn.CylinderComponent.Collisionheight;           
        
        SetupBallShot(v);
       
        return true;
    }     
    
    return false;
}

//check and see if bot will reset ball to map's global kill zone.
//this is a global plane all map's have which kills anyone falling below it.
function bool CheckKillZReset()
{
    local vector v;
    local float max;

    //see if bot near home goal will reset ball by shooting it into nearby killzone    
    if ((HorizontalDistanceFromFriendlyGoal > BallResetDist) || 
        (GetBall(Bot.Pawn) == none) || (! OkToResetBall()))  
    {
        return false;
    }  
          
    CheckForZSurface(vector(Bot.Pawn.Rotation), 1000.0, Bot.Pawn.Location.Z - WorldInfo.KillZ);
    KillZResetDist = ZSurfaceBreakDist;    
    
    if (KillZResetDist <= 0)     
    {
        return false;
    }  
    
    if (! CheckTrace(CLEAR_BALL_RESET_KILLZ_PATH))
    {
        return false;
    }
    
    v = vector(Bot.Pawn.Rotation);
    v.z = 0;
    v = normal(v);
    v = Bot.Pawn.GetPawnViewLocation() + (v * KillZResetDist);

    if (PrimeLauncher())
    {
        return false;
    }      

    //find sharpest angle to shoot ball without bouncing off floor 
    
    max = v.z;
    v.z -= 100;

    while (v.z < (max - 10))
    {
        v.z += 10;
        
        if (BRGame.BlockedDist(Bot.Pawn, normal(v - Bot.Pawn.GetPawnViewLocation()), 
            DistanceBetweenPoints(Bot.Pawn.GetPawnViewLocation(), v) + 30, 
            Bot.Pawn.GetPawnViewLocation(), 
            vect(30,30,30)) == 0)
            {
                break;
            }
    }
    
    if ((! IsLowGravity()) && (KillZResetDist > 100) && (KillZResetDist <= 500))
    {
       //ball lobs up a little when shot so can adjust down a little more for a sharper angle at closer distances.
       //need sharpest angle possible without bouncing on floor so bot can make reset even
       //in tight places with little shooting room.
       v.z -= 20;
    }
                     
    SetupBallShot(v);
    
    ShowPath("Peforming KillZ ball reset");
    
    ResetBallTime = WorldInfo.TimeSeconds;
    TheBall.Resetting = true;

    return true;  
}

//return true if is under heavy attack if health is low and dropping
function bool IsUnderStress()
{   
    return (Bot.Pawn.Health < 50) && (OldHealth > Bot.Pawn.Health);   
}

//return true if bot is under a threat
function bool IsThreatened()
{   
    if (IsUnderStress())
    {
        return true;
    }
    
    return 
      ((WorldInfo.TimeSeconds - Bot.LastSeenTime) <= 2) &&
      (UTBRGame(WorldInfo.Game).IsInstagib || (Bot.Pawn.Health < OldHealth));
}

//check if bot will pass to another player
function bool CheckPlayerPass()
{
    local Controller C;
    local bool ok;
      
    if (AvoidPassing())
    {
        return false;
    }
    
    foreach WorldInfo.AllControllers(class'Controller', C)
    {            
        if (C.Pawn == none)
        {
            continue;
        }
                
        if ((UTVehicle(C.pawn) != none) && (! UTVehicle(C.pawn).bCanCarryFlag))
        {
            continue;
        }
        
        if ((UTBot(C) == none) && (PlayerController(C) == none))
        {
            continue;
        }
        
        //check to see if bot will pass to teammate.
        //they will pass if teammate is closer to base and they got a clear shot at them.           
        //this logic is idiot proofed against the way in prior versions of ut bots used to 
        //pass backwards and disrupt their teams advance, as here bots will only pass towards the enemy base.
        //also will pass to teammate regardless of teammate's position if teammate has issued Cover Me command to bot.
        
        ok = false;

        //teammate closer to enemy base        
        if (
            (C.Pawn.IsSameTeam(Bot.Pawn)) &&
            (HorizontalDistanceBetween(Bot.Pawn, Squad.EnemyGoal) > HorizontalDistanceBetween(C.Pawn, Squad.EnemyGoal)) &&
            Bot.LineOfSightTo(C.Pawn) && 
            (! Bot.Pawn.NeedToTurn(C.Pawn.Location))
           )
        {
            ok = true;
        }
        
        //covering teammate
        if (
            (FormationCenter == C.Pawn) &&       
            (C.Pawn.IsSameTeam(Bot.Pawn)) &&
            Bot.LineOfSightTo(C.Pawn)
           )
        {
            SetFocus(FormationCenter);
            
            if (! Bot.Pawn.NeedToTurn(C.Pawn.Location))
            {
                ok = true;
            }
        } 
        
        if (ok)
        {
            ShowPath("Pass to teammate"); 
                    
            if (PrimeLauncher())
            {
                return false;
            }          
            
            ShootBall(0, true);              
            
            return true;
        }
        
        //check to see if bot will be tricky and attempt to pass ball to enemy so they can frag them and get ball back.
        //they will only do this if doing so would take ball closer to enemy base, as passing backwards would not be wise.
        if (
            (! C.Pawn.IsSameTeam(Bot.Pawn)) &&
            (VSize(Bot.Pawn.Location - C.Pawn.Location) < 500) &&
            (Squad.DistanceFromEnemyGoal(Bot.Pawn) > Squad.DistanceFromEnemyGoal(C.Pawn)) &&
            Bot.LineOfSightTo(C.Pawn) && (! Bot.Pawn.NeedToTurn(C.Pawn.Location))
           )
        {
            if (PrimeLauncher())
            {
                return false;
            }
            
            ShowPath("Pass to enemy");            
                    
            ShootBall();   
            
            return true;
        }       
    }
    
    return false;
}

//return true if bot is allowed to reset ball
function bool OkToResetBall()
{
    //on open maps it's better to lob the ball away than to reset.
    //ball is likely to get further away and nearer to enemy goal.
    
    return 
       ((WorldInfo.TimeSeconds - ResetBallTime) > 30) &&     
       (! CheckTrace(CAN_SEE, Squad.EnemyGoal)) &&
       (! CheckTrace(CAN_SEE, TheBall.HomeBase));
}



//timer used to execute special bot tactics.
//very fine grained timer for precise bot control.
function BotTacticTimer()
{
    local bool didOne, doNav;     
        
    if (WorldInfo.Game.bGameEnded)
    {
        ClearBotTacticTimer();
    }
    
    if ((Bot == none) || (Bot.Pawn == none))
    {
        ClearBotTacticTimer();
        return;
    }
    
    SetVars();
       
    if (ShootingGoal)
    {
        didOne = true;
        ShootGoalTimer();
    }
    
    if (SpecialNavigating)
    {
        didOne = true;
        doNav = true;
        
        if (UTVehicle(Bot.Pawn) != none)
        {
            doNav = false;

            if ((WorldInfo.TimeSeconds - VehicleSpecialNavigationTime) >= 0.20)
            {
                //when special navigating in vehicle not such a fine grained timer is needed,
                //and doing so much processing would be too expensive.
                //jumping is the main thing that needs a very fine grained timer to make the flight path accurate.
                doNav = true;
                VehicleSpecialNavigationTime = WorldInfo.TimeSeconds;
            }
        }
        
        if (doNav)
        {
            SpecialNavigate();
        }
    }
           
    if (DoingBallReset)
    {
        didOne = true;  
        DoingBallResetTimer();
    }    
    
    if (AttackBallBoosting || DefenseBallBoosting)
    {
        didOne = true;
        BallBoostTimer();
    }
    
    if (SuppressEnemy)
    {
        //make bot not kill for 2 seconds after spawning. people 
        //have complained about bots killing too much. even
        //Average bots have GodLike abilities for a short time after
        //spawning.
        
        if (WorldInfo.TimeSeconds <= SuppressEnemyTime)
        {
            Bot.Enemy = none;    
        }
        else
        {
            SuppressEnemy = false;
            Squad.FindNewEnemyFor(Bot, False);
        }
        
        didOne = true;        
    }
    
    if (SpecialAiming)
    {
        didOne = true;
        AimTimer();
    } 
       
    if (! didOne)
    {
        ClearBotTacticTimer();
    }   
}

//called when bot is positioning ball launcher to shoot at something
function AimTimer()
{
    local bool finished;
    local float dist;
    
    PrimeLauncher();
    SetFocalPoint(AimFocalPoint);
    
    if ((GetBall(Bot.Pawn) == none) || (UTBRBallLauncher(Bot.Pawn.Weapon) == none))
    {
        ShowPath("Aborting AimTimer");
        SpecialAiming = false;
        return;
    }
    
    if ((WorldInfo.TimeSeconds - AimTime) >= 0.5)
    {
        finished = true;
    }
    
    dist = GetAimOffset(AimFocalPoint);
      
    //aim is typically right on the first call to this timer at dist 30 - 40,
    //and even if we kept calling this timer, accuracy won't increase.
    if (dist <= 40)
    {
        finished = true;
    }
       
    if (finished)
    {
        ShowPath("Shooting ball in AimTimer. Aim dist from focal point=" $ dist $ " Elasped time=" $ string(WorldInfo.TimeSeconds - AimTime));
        ShootBall();
        SpecialAiming = false;        
    }
}

//return how much distance away from target ball launcher aim is 
function float GetAimOffset(Vector target)
{
    local Vector startLoc, v;
    
    startLoc = Bot.Pawn.GetPawnViewLocation();
    v =  startLoc + (vector(Bot.Pawn.GetViewRotation()) * DistanceBetweenPoints(startLoc, target));
    
    return DistanceBetweenPoints(v, target);
}

//bot shooting ball at goal will attempt to boost ball.
//a defending bot will boost ball away which has been shot by attacker.
function BallBoostTimer()
{
    local bool doBoost;
    local bool finished;
         
    finished = false;
    
    if ((TheBall == none) || (TheBall.HasHitWall) || 
         TheBall.IsHeld() || TheBall.bHome ||
         ((! IsBoostWeapon(Bot.Pawn.Weapon)) && (! HasBoostWeapon(true))) 
       )
    {
        finished = true;
    }

    if (DefenseBallBoosting && (TheBall.TryingToDeflect == none))
    {
        //bot will only hit the ball once to deflect
        finished = true;
    }
    
    if (finished)
    {
        AttackBallBoosting = false;
        DefenseBallBoosting = false;
        Bot.Focus = none;
        Bot.StopFiring();      
        return;
    }
    
    SetFocus(TheBall);
     
    doBoost = false;
   
    if (AttackBallBoosting &&
        CheckTrace(SHOOT_BALL) &&
        ((GoalTarget.Location.Z - TheBall.Position().Location.Z) > -50)  
       )
    {
        doBoost = true;
    }
   
    //not used at moment. use NeedToTurn instead.
    //don't want bot accidentally boosting in enemy ball for their score.
    //in testing NeedToTurn performed better.
    //(! CheckTrace(SHOOT_GOAL,FriendlyGoal)))
        
    if (
        DefenseBallBoosting && 
        Bot.Pawn.NeedToTurn(Squad.FriendlyGoal.Location) &&
        (WorldInfo.TimeSeconds > SuppressBoostTime) &&
        (GetAimOffset(TheBall.Location) <= 50)
        )  
    {
        doBoost = true;
    }      
        
    if (doBoost)
    {      
        //this is the most reliable way to fire the weapon. tried other ways
        //but bot so often the bot wouldn't fire when it should.
        //even with this code, bot won't always fire now, it depends on 
        //if shot lined up well or not.
        Bot.Pawn.Weapon.StartFire(0);  
    }
       
}

//executed by BotTacticTimer to make shoot ball into goal.
//bot will wait until shot is lined up and then shoot.
function ShootGoalTimer()
{
    if (((WorldInfo.TimeSeconds - BotShootTime) > 3) || (GetBall(Bot.Pawn) == none))
    {
        //if 3 seconds pass without being able to shoot, abort attempt
        BotShootTime = 0;
        ShootingGoal = false;
  
        return;
    }  
    
    PrimeLauncher();
    
    SetFocalPoint(GoalTarget.Location);

    //if not lined up shot yet, then keep focusing on target
    if (! CheckTrace(SHOOT_GOAL, GoalTarget))
    {
       return;
    }
       
    ShootBallAtGoal(GoalTarget);
}

//bot is positioning to reset the ball
function DoingBallResetTimer()
{
    local float minDist;
    local bool timedOut;
     
    timedOut = (WorldInfo.TimeSeconds - ResetBallTime) > 10;
    
    if (timedOut)
    {
        CanResetBall = false;
    }
    
    if (
        (TheBall.LastHolder != Bot.Pawn) || TheBall.bHome || 
        (DistanceBetween(TheBall, Squad.FriendlyGoal) > 4000) ||
         timedOut
       )
    {
        DoingBallReset = false;
        return;    
    }

    PrimeLauncher();
        
    if (GetBall(Bot.Pawn) != none)
    {
        SetFocalPoint(Squad.FriendlyGoalInfo.BallResetTarget.Location);
    }
    
    minDist = 1600;
    if (IsLowGravity())
    {
        minDist = 2500;
    }
 
    if ((GetBall(Bot.Pawn) != none) &&
        (! Bot.Pawn.NeedToTurn(Squad.FriendlyGoalInfo.BallResetTarget.Location)) &&
        CheckTrace(SHOOT_RESET) &&
        (DistanceBetween(Bot.Pawn, Squad.FriendlyGoalInfo.BallResetTarget) <= minDist))
    {
        DoingBallReset = false;
        BallLauncher.TheBall.Resetting = true;        
        ShowPath("Performing ball reset");   
        ShootBall();
        Squad.SeekObjective(self);
    }
}


//timer which runs the enhanced spectator camera
simulated function SpectatorTimer()
{
    if ((LocalPlayerController != none) && LocalPlayerController.PlayerReplicationInfo.bOnlySpectator)
    {
        CalcCameraForSpectator();
    }
    else
    {
        ClearTimer('SpectatorTimer');
    }         	
}

/*
  enhanced spectator camera.
  rotates with the viewed player and allows specataor to position the camera 
  exactly where they want. camera will retain it's relative position and rotation
  without the need for the spectator to readjust camera position.
*/
simulated function CalcCameraForSpectator()
{
    local PlayerController PlayerOwner;
    local Actor ViewTarget;
    local vector CamLoc;
    local rotator CamRot;
    local rotator tempRot;
    local Actor temp;    
    
       
    PlayerOwner = LocalPlayerController;
    if (PlayerOwner == none)
    {
        return;
    }

    if (SpectatorFreeCamera && (LastViewTarget == none))
    {     
        PlayerOwner.SetLocation(UTBRGame(WorldInfo.Game).BallHomeBase.Location + vect(-300,0,100));
        PlayerOwner.SetRotation(UTBRGame(WorldInfo.Game).BallHomeBase.Rotation);
        PlayerOwner.ServerSetSpectatorLocation(PlayerOwner.Location);                        
    }
        
    if (SpectatorFreeCamera)
    {       
        PlayerOwner.SetViewTarget(PlayerOwner);
        SpectatorViewTarget = PlayerOwner;
        LastViewTarget = PlayerOwner;
    }

    if ((UTPawn(SpectatorViewTarget) != none) && (SpectatorViewTargetPRI != none) &&
       (UTPawn(SpectatorViewTarget).Health <= 0))
    {
        //this block needed for instant action. in net play when pawn dies it resets to new pawn
        //automatically, but that doesn't happen in instant action so we have to reset it manually.
         
        temp = SpectatorViewTarget;
        PlayerOwner.SetViewTarget(SpectatorViewTargetPRI);    
        SpectatorViewTarget = PlayerOwner.ViewTarget;
        if (PlayerReplicationInfo(SpectatorViewTarget) != none)
        {
           SpectatorViewTarget = temp;
           PlayerOwner.SetViewTarget(PlayerOwner);
        }
    }
           
    if ((PlayerOwner.ViewTarget != none) && (PlayerOwner.ViewTarget != PlayerOwner))
    {
        SpectatorViewTarget = PlayerOwner.ViewTarget;
        PlayerOwner.SetViewTarget(PlayerOwner);
    }
    
    if (SpectatorViewTarget == none)
    {
        return;
    }
        
    if (SpectatorViewTarget == PlayerOwner)
    {       
        return;
    }
      
    ViewTarget = SpectatorViewTarget;

    if (Vehicle(ViewTarget) != none)
    {
        ViewTarget = Vehicle(ViewTarget).Driver;
    }
    
    if (ViewTarget == none)
    {
       return;
    }
 
           
    if ((ViewTarget != LastViewTarget) || (LastViewTarget == none))
    {
        if (UTBRGoal(ViewTarget) != none)
        {
            CameraOffset = Vect(0,200,0);
            CameraRotationOffset = -16384;                    
        }
        else
        {
            CameraOffset = Vect(-200,0,50);
            CameraRotationOffset = 0;            
        }
              
        CamLoc = ViewTarget.Location + (CameraOffset >> ViewTarget.Rotation);
        LastViewTargetRotation = ViewTarget.Rotation;        
        
        PlayerOwner.SetLocation(CamLoc);

        tempRot = ViewTarget.Rotation;
        tempRot.Pitch = 0;
        tempRot.Roll = 0;
        PlayerOwner.SetRotation(tempRot);                   
        
        CameraLastRotation = PlayerOwner.Rotation;        
        CameraLastLocation = PlayerOwner.Location;        
    }
      
    CameraOffset = 
       ((Normal(PlayerOwner.Location  - CameraLastLocation) * 
        vsize(CameraLastLocation - PlayerOwner.Location)) << LastViewTargetRotation) 
        + CameraOffset;
                                      
    CamLoc = ViewTarget.Location + (CameraOffset >> ViewTarget.Rotation);  
    LastViewTargetRotation = ViewTarget.Rotation;
            
    CamRot = PlayerOwner.Rotation;
    
    CameraRotationOffset = 
        (PlayerOwner.Rotation.Yaw - CameraLastRotation.Yaw) + 
        CameraRotationOffset;
    CamRot.Yaw = ViewTarget.Rotation.Yaw + CameraRotationOffset;

    CameraLastRotation =  CamRot;
    PlayerOwner.SetRotation(CamRot);
    
    CameraLastLocation = CamLoc;
    LastViewTarget = ViewTarget;


    PlayerOwner.SetLocation(CamLoc);
    PlayerOwner.ServerSetSpectatorLocation(Camloc);

	PlayerOwner.bCollideWorld = false;
    PlayerOwner.SetCollision(false, false);
}

//for debugging. show path bot is taking.
function ShowPath(string msg)
{
    BRGame.SendPlayerDebugMessage(Controller, msg);
}

//return how high a1 is over a2, negative if below. 
function float VerticalDistanceBetween(Actor a1, Actor a2)
{
    return UTBRGame(WorldInfo.Game).VerticalDistanceBetween(a1, a2);    
}

//return the horizontal distance a1 is away from a2.
function float HorizontalDistanceBetween(Actor a1, Actor a2)
{   
    return UTBRGame(WorldInfo.Game).HorizontalDistanceBetween(a1, a2);
}

//return the horizontal distance v1 is away from v2.
function float HorizontalDistanceBetweenPoints(vector v1, vector v2)
{
    return UTBRGame(WorldInfo.Game).HorizontalDistanceBetweenPoints(v1, v2);      
}

//return how far away a1 is from a2
function float DistanceBetween(Actor a1, Actor a2)
{
    return UTBRGame(WorldInfo.Game).DistanceBetween(a1, a2);
}

//return how far away a1 is from a2
function float DistanceBetweenPoints(vector v1, vector v2)
{  
    return UTBRGame(WorldInfo.Game).DistanceBetweenPoints(v1, v2);
}


/*
draw trace to enemy base based on weapon aimed position, or player view.
return true if trace can find enemy base without being blocked by a blocking actor.
trace will ignore non blocking actors.

operation=CAN_SEE : return true if owner can see the target.
the ut function LineOfSightTo() will not work for this because there are actors
which are not blocking, meaning you can pass through them, but
still are visible are cause LineOfSightTo to return false. other actors
are blocking but visually you can see through them but we don't want to consider that 
as 'seeing'. it should be a clear line of sight.

operation=SHOOT_GOAL : return true if owner has a clear shot at goal.

operation=CLEAR_SHOT_PATH : return true if ball launcher has a clear path to pass ball forward

operation=SHOOT_BALL : return true if owner has a clear shot at boosting ball.

operation=SHOOT_RESET : return true if owner has a clear shot at resetting ball.

operation=CLEAR_BALL_RESET_KILLZ_PATH : return true if owner has a clear shot a reseting ball to map's global kill zone.
 
*/
function bool CheckTrace(int operation, optional Actor target)
{
    local vector HitLocation, HitNormal, StartTrace, EndTrace;
    local actor actor;
    local float allowedDist;
    local Vector extent;
    local float hitDist;
      
    //GetPhysicalFireStartLoc is the best starting place, as this is where the ball would
    //spawn when fired. InstantFireStartTrace is not too good as for a bot it's just the pawn location.
    //GetPawnViewLocation is a bit better as that's the eye location.
    //StartTrace = BallLauncher.InstantFireStartTrace();
    //StartTrace = Instigator.GetPawnViewLocation();
    StartTrace = BallLauncher.GetPhysicalFireStartLoc();
    
    //EndTrace = BallLauncher.InstantFireEndTrace(StartTrace);
    EndTrace = StartTrace + (vector(Bot.Pawn.GetViewRotation()) * BallLauncher.GetTraceRange());    

    extent = vect(0,0,0);
    
        
    if (operation == CAN_SEE)
    {
        StartTrace = Bot.Pawn.GetPawnViewLocation();    
        EndTrace = target.Location;
    }

    if (operation == SHOOT_RESET)
    {
        extent = vect(30,30,30);
        
        if (UTBRGoal(Squad.FriendlyGoalInfo.BallResetTarget) != none)
        {
            operation = SHOOT_GOAL;
            target = Squad.FriendlyGoalInfo.BallResetTarget;
        }        
    }
    
    if (operation == SHOOT_GOAL)
    {
       //35 does not work well.
       //in regular gravity in VoidyTempleBurner at the apex of double jump bot should be
       //able to shoot and score. extent=35 will fail the trace and thus bot
       //won't try shot, but 30 will succeed.
       //so, 30 is the max extent we want. don't want to make it too small though or
       //bot will try shot and miss because ball will bounce off rim.     
       extent = vect(30,30,30);  
    } 
       
    if (operation == CLEAR_SHOT_PATH)
    {
       //check if bot has clear path ahead for a short pass
       extent = vect(30,30,30);
       StartTrace = Bot.Pawn.GetPawnViewLocation();     
       EndTrace = StartTrace + (vector(Bot.Pawn.Rotation) * 2000);
    }

    if (operation == CLEAR_BALL_RESET_KILLZ_PATH)
    {
       //check if bot has clear path ahead to reset ball to map's global killzone
       extent = vect(30,30,30);
       StartTrace = Bot.Pawn.GetPawnViewLocation();     
       EndTrace = StartTrace + (vector(Bot.Pawn.Rotation) * KillZResetDist);
    }
          
    //Trace() will not work here. it does not find a UTBRGoal because it is a non blocking actor.
    //can not use UTWeapon's CalcWeaponFire because it uses Trace().
    //TraceComponent() will not work because it can't tell if another actor blocks the trace or not.
            
    foreach BallLauncher.GetTraceOwner().TraceActors( class'Actor', actor, HitLocation, HitNormal, EndTrace, StartTrace, extent,,)
    {   
        hitDist = vsize(Bot.Pawn.Location - HitLocation);

        /*
           note on BlockingVolumes. they are invisible and block players and ball.
           annoying for ball but unavoidable. they are used for meshes which have
           no collision so that players can be blocked. projectiles don't have this
           problem because they collide at poly level with mesh. projectiles use
           zero extent collision which means they only collide at their center point.
           the volumes let zero colliders pass through. ball can not be a zero collider
           because we need whole ball to collide, and also ball would sink into floor up
           to it's center and then could not be picked up.
        */
        
        if (ForcedDirVolume(actor) != none)                 
        {
           //stange actors which have bBlockActors=true but dont' actually block 
           continue;
        }                

        if (operation == CLEAR_BALL_RESET_KILLZ_PATH)
        {              
            if (actor.bBlockActors)
            {          
                return false;                
            }
                    
            continue;
        }
                
        if (operation == CLEAR_SHOT_PATH)
        {       
            if (actor.bBlockActors)
            {
                allowedDist = 1200;
                if (IsLowGravity())
                {
                    allowedDist = 2000;
                }
                
                if (hitDist < alloweddist)
                {
                    return false;
                }                
                               
                return true;
            }
                    
            continue;
        }     
            
        if ((operation == CAN_SEE) || (operation == SHOOT_GOAL))
        {
            //ball test used for ball boosting. for example, don't want defender to accidentally
            //boost ball into goal. trace should go through ball to see if aim
            //is lined up with goal or not. don't defense boost if lined up.                                 
            if ((UTBRGoal(target) != none) && (UTBRBall(actor) != none))
            {
               continue;
            }

            if (actor == target)
            {     
              return true;
            }
                        
            if (actor.bBlockActors)
            {
              return false;
            }
        }        
        
        if (operation == SHOOT_RESET)
        {
            if ((vsize(HitLocation - Squad.FriendlyGoalInfo.BallResetTarget.Location) <= 800) ||
                (actor == Squad.FriendlyGoalInfo.BallResetTarget))          
            {
                return true;
            }
            
            if (actor.bBlockActors)
            {
              return false;
            }
        }                          
        
        if (operation == SHOOT_BALL)
        {
            if (actor.isa('UTBRBall'))
            {
                return true;
            }
            
            if (actor.bBlockActors)
            {
              return false;
            }
        }
    }
    
    if ((operation == CLEAR_SHOT_PATH) || (operation == CLEAR_BALL_RESET_KILLZ_PATH))
    {
        return true;
    }
    
    return false;
}

//ut3 makes bot do single jump off ball if bot lands on sitting ball. very annoying.
//in low grav bot can bounce off ball way up and then go back down which wastes a lot of time.
//probably due to a bump being reported when bot hits ball. ut3 has code somewhere
//(maybe in native engine) which makes bot jump when bumping mesh in certain situations.
//when landing on ball we want to continue landing not go up, so
//this routine is used to make sure bot's movetarget will keep bot landing.
//return true if should reject target due to landing on ball and target is too high.
function bool LandedOnBallRejection(Actor target)
{
    if (! LandedOnBall)
    {
        return false;
    }
    
    if (Bot.Pawn.Physics != PHYS_Falling)
    {
        LandedOnBall = false;
        return false;
    }
    
    if ((target == none) || DefensivePositionFailed)
    {
        //if all pathing failing we have no choice but to let bot route at high target
        return false;
    }
    
    if ((target.Location.Z - Bot.Pawn.Location.Z) > 50)
    {
        return true;
    }
    
    return false;
}

//return true if bot can see it's objective.
//unlike the ut LineOfSightTo function, this will not reporting seeing
//when bot is looking through a blocking but see-through material like a fence.
//important because ai logic is based on bot's seeing something and a good line of
//sight should mean there's a clear unblocked path along that line.
//we use LineOfSightTo when this is not a critical issue, as this function will take
//more processing time.
function bool CanSeeObjective()
{       
    return CheckTrace(CAN_SEE, FormationCenterTarget);
}

//return true if can see target
function bool CanSee(Actor target)
{
    if (target == FormationCenter)
    {
        return CanSeeObjective();
    }
    
    return CheckTrace(CAN_SEE, target);
}

//try to perform special navigation (dynamic pathing) to target.
//return true if navigation started. 
function bool TrySpecialNavigation(Actor dest, bool force)
{
    local bool didJump;
       
    if (LandedOnBallRejection(dest))
    {
        ShowPath("Not special navigating: Landed on ball");
        return false;
    }
    
    if ((! force) && SpecialNavigationNotGood(dest))
    {
        ShowPath("Not special navigating: " $ Reason);
        
        return false;
    }

    InitSpecialNavigation(false, true);
    SpecialNavigationPathTarget = dest;
    
    if (dest == PathTarget)
    {
        TriedForcedSpecialNavigationToPathTarget = true;
    }
           
    if (SeekDirect && IsFormationCenter(dest))
    {
        SetPathTarget(dest, true);
    }    

    if (CanInitiateJump() && IsSoaking && (! ForcedJumpInSoak))
    {
        ForceJump = true;
        ForcedJumpInSoak = true;            
    }
    
    didJump = false;
            
    if (CanInitiateJump() && (JumpingPrefered(dest) || ForceJump))
    {
        ForceJump = false;
        didJump = TrySpecialJump(dest);
        
        /*
        when bot hits a blockage such as a pillar pathing will break and so this
        code is run which then uses jumps and the jump navigation in SpecialNavigate, which 
        has advanced navigation to get around. the height and added mobility in a jump
        are ideal for getting by a lot a sticky situations.

                
        note on bot MoveToward function. below code will call MoveToward and
        does make bot move to target, but it's horrible. if bot is blocked by
        wall they just keep walking against wall and don't strafe around.
                
        Bot.MoveTarget = dest;
        Bot.GotoState('Defending', 'Moving');
        ShowPath("MOVETO " $ dest);


        below works better but not very well. bot commonly goes straight backwards
        when hitting wall. sometimes it will strafe to side but when clear will 
        not keep moving towards target, ending up strafing into nearby ledge and jumping off.
        tested in ghost mode while holding ball and hiding behind pillar, and once bot started
        move towards me then i'd move to another location but bot would not follow me and
        just keep moving to pillar, then hit pillar and start moving backwards.
                
        Bot.Destination = dest.Location;
        Bot.Enemy = none;
        Bot.GotoState('FindAir', 'DoMove');
        ShowPath("MOVETO " $ dest);
        */
    }
    
    if (! didJump)    
    {
        SetMoveTarget(dest);
        
        if (! SpecialNavigating)
        {             
            StartSpecialNavigation(dest);
        }
        else
        {
            ShowPath("Routing special navigation to forced target " $ dest);            
        }
    }
                
    if (SpecialNavigating)
    {
        ForceRepath = false;       
    }
    else
    {
        SpecialNavigationPathTarget = none;
    }
    
    return SpecialNavigating;
}

/*
bot will jump up to target.

useful for when goal is very high, otherwise normal ut pathing would suffice to reach goal.
ut pathing can't reach places high in the air.
this routine does many things to make bot air navigate properly to a goal, which ut does not do for us.

needed for jump pad support, as normal ut doesn't handle this too well.
in ut2004 off a jump pad near the goal a bot would go for a shootspot instead of going to goal.

when bot did manage to try to jump towards a goal they would rarely hit because ut does not
adjust the bot's acceleration during flight to steer them to the goal.

with regular ut jumping they never try a jump at a target until they are in about 1/2 the range of a single
jump, so the bot would never try a long jump towards a target, which is really silly in low gravity.

for goals which are high in the air and so nav pathing can't find them, bot will always try to
get to the closest nav point and try the jump from there instead of just trying to jump directly at the goal.

regular ut jumping does not take into account quad jumping, so bot only tries a double jump.

bots with low skill level tend to freeze up because they don't know tricks like trying to jump up to
a higher point first and then trying to jump at goal.

*/
function bool TryJump(Actor target)
{
    local vector temp;
        
    if (! CanInitiateJump())
    {
        return false;
    }
         
    temp = target.Location;
    if (ZStrafing)
    {
        temp = StrafeTarget;
    }

    TurningForGoalJumpShot = false;
                  
    Bot.Pawn.Velocity = Normal(temp - Bot.Pawn.Location) * Bot.Pawn.GroundSpeed;
    Bot.Pawn.Velocity.Z = 0.f;
        
    Bot.Pawn.Acceleration = vect(0,0,0);
    
    Bot.Pawn.DoJump(false);
    JumpStartedByPlayerManager = true;

    ShowPath("Trying jump at " $ target);
    
    StartSpecialNavigation(target);
    
    return true;
}

function bool CanInitiateJump()
{
    if (UTBRGame(WorldInfo.Game).Test_NoJump)
    {
        return false;
    }
        
    if (Bot.Pawn.Physics != PHYS_Walking)
    {
        return false;
    }
    
    return true;
}

//take at jump at the enemy goal
function bool TryGoalJump(UTBRGoal goal)
{
    local float zDist;
    local bool findPosition;

    if (! CanInitiateJump())
    {
        return false;
    }
           
    GoalTarget = goal;
          
    zDist = goal.Location.Z - Bot.Pawn.Location.Z;
    
    //below BlockedDist checks to make sure the place where bot would jump from is clear of blockages
    //important to not pass in an extent as we just want to check for blockage like a wall
    //and an extent can collide with a wide collision area around some goal rims, but a zero extent
    //is more likely to get through and will also pass through things like blocking volumes.
    
    FindGoodGoalJumpPosition(goal);
    findPosition =
        (! JumpingOverBlockage) && 
        (! CanJumpTo(goal)) && 
        GoodFloorUpTo(GoalJumpPosition) &&
        (HorizontalDistanceBetween(Bot.Pawn, goal) <= 1000) &&
        (! bool(UTBRGame(WorldInfo.Game).Settings.DisableShootingScore)) &&
        (zDist > 100) &&
        (BRGame.BlockedDist(self, Normal(goal.Location - GoalJumpPosition), DistanceBetweenPoints(goal.Location, GoalJumpPosition) - 100, GoalJumpPosition) == 0);
        ;
       
    if (findPosition)
    { 
        //find best position to jump for goal
        //bot is turning their gun towards the goal before they try jumping at goal. 
        //if they miss goal and start to go down, they may try a shot which is why 
        //they should be pointed at goal first.        
        
        TryJumpShot = true;
        DistFromDesiredPosition = 999999999.0;
        TurningForGoalJumpShot = true;
        SpecialNavigationTime = WorldInfo.TimeSeconds;
        ShowPath("Setup goal jump shot. zDist=" $ zDist);        
        StartSpecialNavigation(goal);       
    }      
    else
    {
        ShowPath("Goal jumping");
            
        if (! TryJump(goal))
        {
            return false;
        }
    
        //we are near goal and paths are broken due to high goal,
        //so if jump fails bot will try a shot
        if ((NoGoalPath) && 
            (HorizontalDistanceBetween(Bot.pawn, goal) <= 1000) &&
            (GetBall(Bot.Pawn) != none) &&
            (! bool(UTBRGame(WorldInfo.Game).Settings.DisableShootingScore))
            )
        {
            TryJumpShot = true;
        }
               
    }
                  
    GoalJumping = true;
    if (GetBall(Bot.pawn) != none)
    {
        GoalJumpingWithBall = true;    
    }
    
    return true;
}

//perform forward dodge as a means of faster travel
function PerformForwardDodge(Actor target)
{
    local vector Dir;
    
    ShowPath("Perform forward dodge at " $ target);
    
    SuppressLoopChecks = true;
                 
    SetMoveTarget(target);
    
    Dir = normal(MoveTarget.Location - Bot.Pawn.Location);
    Dir.Z = 0;
    Dir = Normal(Dir);

    UTPawn(Bot.Pawn).CurrentDir = DCLICK_Forward;     
    UTPawn(Bot.Pawn).PerformDodge(UTPawn(Bot.Pawn).CurrentDir, Dir,Normal(Dir Cross vect(0,0,1)));
}


//try jump at target using special moves, such as jumping goal shot at goal, etc
function bool TrySpecialJump(Actor dest)
{
    if (UTBRGoal(dest) != none)
    {
        return TryGoalJump(UTBRGoal(dest));
    }
   
    return TryJump(dest);        
}

//return true if bot needs to jump to reach target
function bool NeedsJump(Actor target)
{
    if (target == none)
    {
        return false;
    }
    
    return NeedsJumpToPoint(target.Location);
}

function bool NeedsJumpToPoint(Vector target)
{
    return (target.Z - Bot.Pawn.Location.Z) >= 50;
}

//return true if bot should not jump to reach target
function bool JumpingNotGood(Actor target)
{
    local bool ok;

    //note: for high goals bot will performing jumping goal shot even if jump can't reach goal.
    
  
    if (! NeedsJump(target))
    {
         Reason = "Target doesn't need jump";
         return true;
    }

    if (CanSee(target) && CanJumpTo(target))
    {
        return false;
    }
                     
    if (IsFormationCenter(target) && ReachedFallback)
    {
        //doesn't look like bot should make jump.
        //however, if close enough to objective then jump if DefensivePosition doesn't look better

        if (CachedDefensivePosition == none)
        {
            //CachedDefensivePosition used as an optimization since this search is expensive 
            CachedDefensivePosition = Squad.FindDefensivePosition(self);
        }

        if (CachedDefensivePosition != none)
        {
            ok = true;
            
            if (CanJumpToHeightOf(target) && 
                (DistanceBetween(Bot.Pawn, target) < DistanceBetween(CachedDefensivePosition, target)))
            {
                //CachedDefensivePosition is not a better place to jump from since bot
                //can jump as high as target and is closer to target than CachedDefensivePosition 
                ok = false;
            }
                                                                    
            if (ok &&
                (HorizontalDistanceBetween(target, CachedDefensivePosition) <= 1000) &&
                ((CachedDefensivePosition.Location.Z - Bot.Pawn.Location.Z) >= 50) &&
                (CanJumpToHeightOf(target, CachedDefensivePosition.location.z)))
            {
                //better place found. don't jump.
                Reason = "Better place to jump from: " $ CachedDefensivePosition;
                return true;
            }
        }

        //no better place found to jump from so allow jump.
        return false;
    }

    Reason = "Not seeking formation center or not reached fallback";
    return true;
}

//return true if bot should jump rather than walk.
//does not call JumpingNotGood because we want to be less critical here of how feasible jump is.
//bot can be special navitating already and just needs to know when it's close enough to try jump.
function bool JumpingPrefered(Actor target)
{
    if (NeedsJump(target))
    {
        if (CloseEnoughToJump(target))
        {
            return true;
        }
        
        
        if ((UTBRGoal(target) != none) &&
            (! CanJumpToHeightOf(target)) &&
            (HorizontalDistanceBetween(Bot.Pawn, target) <= GoalJumpDist(UTBRGoal(target))))
        {
            return true;
        }
    }
    
    return false;
}

//return true if bot should not use special navigation to reach target
function bool SpecialNavigationNotGood(Actor target)
{
     Reason = "";
     
     if (SeekDirect && IsFormationCenter(target))
     {
         if (! ReachedFallback)
         {
             //bot too far away
             Reason = "Not reached fallback";  
             return true;
         }

         if (UTVehicle(Bot.Pawn) != none)
         {
             if ((VerticalDistanceFromFormationCenter > 100) && 
                 (! UTVehicle(Bot.Pawn).bCanFly))
             {
                 Reason = "Vehicle can't fly to";
                 return true;
             }
         }
         else
         {
             if (NeedsJump(target) && (! CanJumpToHeightOf(target)) && JumpingNotGood(target))
             {
                 return true;
             }
         }
     }
     
     //ok to use special navigation
     return false;       
}

//return true if bot should keep moving to target even when reached
function bool KeepMovingWhenReached()
{
    local bool keepMoving;
    
    //keep moving if seeking ball or ball runner seeking goal 
    //because then bot needs to keep moving at target until it gets ball or scores
    
    keepMoving = false;
    if ((UTBRBall(MoveTarget) != none) && (GetBall(Bot) == none))
    {
        keepMoving = true;
    }    
    
    if ((GetBall(Bot.Pawn) != none) && (UTBRGoal(MoveTarget) != none))
    {
        keepMoving = true;
    }
    
    if (TurningForGoalJumpShot)
    {
        keepMoving = true;
    }
    
    if (UTJumpPad(MoveTarget) != none)
    {
        //jumppad automatically switches movetarget, no need to repath
        keepMoving = true;
    }
    
    return keepMoving;
}

//used during special navigation to set ut target info according
//to where special navigating to
function SetUTBotMoveTarget()
{
    Bot.Destination = Destination;
    Bot.MoveTarget = none;

    if (Destination == vect(0,0,0))
    {
        Bot.MoveTarget = MoveTarget;
    }
}

//perform special bot navigation. started by calling StartSpecialNavigation.
function SpecialNavigate()
{
    local bool finished;
    local float hDist, zDist, hVel, velAdj, dist;
    local vector temp, target, startLoc, tempDest;
    local Actor targetActor;
    local bool slowDown, accelerate, checkShot;
    local int zDir;
    local UTBRBall ball;
    local Actor focusOn;
             
    /*
    ut can only make a bot jump into a goal near to ground.
    goals high up need this special logic to guide the bot there.
    ut can not path to goals high in the air.
    also, ut bot air navigation in general is pretty poor, as velocity is
    pointed at target only at the start of jump and then no more navigation occurs
    to guide bot to target.     
    */
            
    if (WorldInfo.Game.bGameEnded)
    {
        finished = true;
    }

    //if finding random target, it will timeout in a few seconds, and we don't want to cancel just because
    //physics changed. it's important to keep bot moving, as at this point, it's in a pretty tight situation.
        
    if (SpecialNavigationPhysics != Bot.Pawn.Physics)
    {
        Reason = "Physics change from " $ SpecialNavigationPhysics $ " to " $ Bot.Pawn.Physics;       
        finished = true;
        
        if ((SpecialNavigationPhysics == PHYS_Walking) && (Bot.Pawn.Physics == PHYS_Falling) && ZStrafing)
        {
            //bot is walking off a ledge heading to zStrafe point just off ledge.
            //sometimes ut has trouble with this and bot wobbles on edge of ledge.
            //keeping bot in special navigation mostly (bot not always) eliminates this.
            //navigation will end due to physics change when ZStrafing goes false.
            finished = false;
        }      
    }
        
    if ((Bot.Pawn == none) || (MoveTarget == none))
    {
       Reason = "Pawn/MoveTarget None";   
       finished = true;
    }
   
    if (MoveTarget == none)
    {
        Reason = "MoveTarget=None";
        finished = true;
    }
   
    if (TurningForGoalJumpShot)
    {
        //line bot up for shot
        //perpendicular to goal is the best way to back up because it will be perfectly lined up for shot.
        //must keep in mind goal can be horizontal or vertical. 
        //this way accounts for both.
                 
        finished = true;
       
        //bot is at the desired place once it's close to it or starts moving away from it
        dist = HorizontalDistanceBetweenPoints(Bot.Pawn.Location, GoalJumpPosition);
        if (dist < DistFromDesiredPosition)
        {
            DistFromDesiredPosition = dist;
        }

        if ((dist <= (DistFromDesiredPosition + 10)) && (dist > 10))
        {
            //because jump will take bot closer to goal we must get far enough away
            //so that shot can make it.
            //if it's a goal facing downward then can take shot when bot's below goal,
            //which is the case when hort dist from GoalJumpPosition is < 100.
            finished = false;              
        }
    
        if ((WorldInfo.TimeSeconds - SpecialNavigationTime) > 5)
        {
           finished = true;
        }
        
        if (GetBall(Bot.Pawn) == none)
        {
            finished = true;
        }
               
        if (finished && (GetBall(Bot.Pawn) != none))
        {
            SpecialNavigating = false;
            
            //after bot has done it's best to get in position, jump at goal
            if (TryJump(GoalTarget))
            {
                TurningForGoalJumpShot = false;
                finished = false;
            }
        }
    }
               
    if ((! KeepMovingWhenReached()) && (! finished) &&
        ReachedTarget(MoveTarget))
    {              
        Reason = "Reached target";
        ForceRepath = true;        
        finished = true;
    }     
           
    if ((! finished) && CanInitiateJump() && (! TurningForGoalJumpShot))
    {            
        if (
            ((MoveTarget == PathTarget) && JumpingPrefered(MoveTarget)) ||
            (ZStrafing && (StrafeTarget.Z > Bot.Pawn.Location.Z))
           )
        {
            //should now stop walking so that a jump can be made
            finished = true;
            Reason = "Force jump";
            ForceJump = true;
        }
    }   
    
    if (FindingRandomTarget)
    {
        if ((WorldInfo.TimeSeconds - SpecialNavigationTime) > 4)
        {
            //have completed all the steps of trying to resolve path and failed.
            //go back to beginning of path cycle and start over.

            Reason = "Done finding random target. Restarting path cycle.";
            finished = true;
            RestartPathCycle();
        }
    }
    
    if (CheckSpecialNavigationMoveTargetChange() && (! finished)) 
    {
        Reason = "Forced MoveTarget change";
        finished = true;            
    }   
    
    if ((! finished) && (! GettingSpecialItem()) && (GetBall(Bot.Pawn) == none) && NeedWeapon() && (! Retasking))
    {
        //Reason = "Need weapon";
        ForceRepath = true;
        //SpecialNavigationPathTarget = none;
        //finished = true;
        
        ShowPath("Retask while special navigating: need weapon");
        Retask();      
    }
       
    if (finished && (SpecialNavigationPhysics == PHYS_Falling) && (Bot.Pawn.Physics == PHYS_Falling))
    {
        //for falling we want to keep navigating as long as in air, so just retask.
        //this is why above we set ForceRepath in a few places, to make sure repathing is done
        //(normally EndSpecialNavigation handles setting ForceRepath).    
        finished = false;

        if (! Retasking)
        {
            ShowPath("Retask while falling: " $ Reason);
            Retask();
        }
    }
    
    CheckJumpPad();   

    if (MoveTarget == none)
    {
        Reason = "MoveTarget=None";
        finished = true;
    }
            
    if (finished)
    {
        EndSpecialNavigation();
        return;
    } 

    if (UTVehicle(Bot.Pawn) == none)
    {
        //without this bot is in state WaitingForLanding, which can override our current move target and velocity,
        //especially by the NotifyMissedJump() call.
        //need a state which does little so we can take control.
        //not using MoveToGoal because MoveToGoalWithEnemy allows bot to shoot at enemy if needed.
        
        Bot.GotoState('MoveToGoalWithEnemy');
   }

    if (
        (Bot.Pawn.Physics == PHYS_Falling) &&
        (UTBRGoal(FormationCenter) != none) &&
        IsSeekingFormationCenter() &&
        (MoveTarget != FormationCenterTarget) &&
        (SeekDirect) &&         
        (DistanceFromFormationCenter <= 1000) &&
        (UTJumpPad(LastMoveTarget) != none) &&
        FacingGeneralDirection(FormationCenterLocation, 0.4)
       )
    {
        //jumppads are usually pointed at some generic pathnode, not the objective, 
        //so steer bot to objective if it's close enough. this makes a bot hit the 
        //goal right off the pad and even use pads which aren't targeted at goal.
        //we don't do this for ball because main pathing might have failed and bot 
        //could be following alternate path and we don't want to disrupt that path.
            
        ShowPath("Rerouted to " $ FormationCenter $ " from jump pad");
        
        SetPathTarget(FormationCenter, true);
    }
          
    LastFormationCenterLocation = FormationCenterLocation;    

    if (! SpecialFocus)
    {
        focusOn = MoveTarget;
        
        if ((UTVehicle(Bot.Pawn) == none) &&
            (GetBall(Bot.Pawn) != none) && (UTBRGoal(MoveTarget) != none))
        {
            //keep ball runner focus on goal so it can pull off a goal shot if needed
        }
        else if ((GetBall(Bot.Pawn) != none) && (! CanUseWeaponsWithBall()))
        {
            //keep ball runner focused on target unless they can use weapons
        }
        else
        {
            //for bot to be able to attack enemy, bot's focus must be on enemy
            
            ball = GetBall(MoveTarget);
            if ((ball != none) && ball.IsHeldByEnemyOf(Bot.Pawn) && Bot.LineOfSightTo(MoveTarget))
            {
                //make bot single out the enemy ball runner to attack
                Bot.Enemy = ball.HolderController.Pawn;
            }
            
            if (
                (Bot.Enemy != none) && Bot.LineOfSightTo(Bot.Enemy) &&
                FacingGeneralDirection(Bot.Enemy.Location) &&
                ((WorldInfo.TimeSeconds - EnemyLoseFocusTime) > 3)
               )
            {
                //if bot can attack enemy and still face movetarget then
                //allow bot to attack while moving 
                focusOn = Bot.Enemy;       
            }
        }
        
        if ((Bot.Focus == Bot.Enemy) && (focusOn != Bot.Enemy))
        {
            EnemyLoseFocusTime = WorldInfo.TimeSeconds;
        }
              
        SetFocus(focusOn);
    }    
           
    hDist = HorizontalDistanceBetween(Bot.Pawn, MoveTarget);
    zDist = VerticalDistanceBetween(MoveTarget, Bot.Pawn);

   
    zDir = 0;
    
    if (LastZ > Bot.Pawn.Location.Z)
    {
       zDir = -1;
    }

    if (LastZ < Bot.Pawn.Location.Z)
    {
       zDir = 1;
    }
    
    LastZ = Bot.Pawn.Location.Z;          
    
    //unlike ut apex event which waits for apex of the current jump, 
    //WaitingForMultiJumpApex waits for the apex of a multi jump
    if ((zDir == -1) && (Bot.Pawn.Physics == PHYS_Falling))
    {
        WaitingForMultiJumpApex = false;
        KeepJumping = false;          
        
        if (DistanceFromFormationCenter > (ClosestDistanceToFormationCenter + 25))
        {  
            //falling away from goal
            checkShot = true;
        }     
    }
    
    if (GetBall(Bot.Pawn) == none)
    {
        TryJumpShot = false;
    }

    //see if bot will shoot ball early because clearly can't make jump but is close enough to shoot
    if ((Bot.Pawn.Physics == PHYS_Falling) &&
        ((Squad.EnemyGoal.Location.Z - JumpStartLocation.Z) > (JumpHeight + 150)) && 
        TryJumpShot && (FormationCenter == Squad.EnemyGoal) &&
        (! CanJumpTo(Squad.EnemyGoal, JumpStartLocation)) &&
        ((TheBall.TravelDistance(self, 0) * 0.8) > DistanceFromFormationCenter))
    {
        //looks like bot is close enough to make the shot
        checkShot = true;
    }

    //below BlockedDist checks for blockage like wall which would prevent shot, or if bot is directly below
    //goal rim. we don't want to do a full ball extent check because that can fail on some goal rims which
    //have very wide collision area (which ball would pass through)
    startLoc = BallLauncher.GetPhysicalFireStartLoc();          
    if (
        checkShot &&
        (UTBRGoal(MoveTarget) != none) && 
        (zDist > 100) && 
        TryJumpShot &&
        (GetAimOffset(UTBRGoal(MoveTarget).Location) <= 35) &&
        (BRGame.BlockedDist(self, Normal(UTBRGoal(MoveTarget).Location - startLoc), DistanceBetweenPoints(UTBRGoal(MoveTarget).Location, startLoc) - 100, startLoc, vect(5,5,5)) == 0)
       )
    {
        //at the apex of jump and still has not reached goal so try goal shot

        TryJumpShot = false;

        //just go ahead and shoot instead of checking ball shot path trace 
        //because for goals with large collision areas traces will fail but shot could still be made
        ShootBallAtGoal(UTBRGoal(MoveTarget));
    }
    
    if (DistanceFromFormationCenter < ClosestDistanceToFormationCenter)
    {
        ClosestDistanceToFormationCenter =  DistanceFromFormationCenter;
    }   
        
    //make bot strafe around blockages
    Destination = MoveTarget.Location;
    if (TurningForGoalJumpShot)
    {
        Destination = GoalJumpPosition;
    }
    if (FindingRandomTarget)
    {
        Destination = RandomDestination;
    }
    
    tempDest = Destination;
    
    target = SuggestStrafeDestination();
    
    if (JumpOverBlockage)
    {
        ForceJump = true;
        JumpOverBlockage = false;
        JumpingOverBlockage = true;
        EndSpecialNavigation();
        return;
    }
    
    targetActor = none;

    if (target == vect(0,0,0))
    {
        target = tempDest;
        
        if (MoveTarget != none)
        {
            targetActor = MoveTarget;
        }
    }
    
    Destination = target;
    
    
    hDist = HorizontalDistanceBetweenPoints(Bot.Pawn.Location, target);
    zDist = target.Z - Bot.Pawn.Location.Z;      

    if (Bot.Pawn.Physics == PHYS_Falling)
    {
        if (OverTarget(target))
        {      
           KeepJumping = false;       
        } 
            
        if (UTPawn(Bot.Pawn) != none)
        {        
            if ((UTPawn(Bot.Pawn).MultiJumpRemaining < 1) || (UTPawn(Bot.Pawn).bDodging))
            {      
               KeepJumping = false;       
            }
        }       
                    
        //make bot keep jumping or stop jumping
        if (KeepJumping)
        {
           Bot.bPendingDoubleJump = true;
           Bot.bNotifyApex = true;
        }
        else
        {
           Bot.bPendingDoubleJump = false;
           Bot.bNotifyApex = false;           
        }      
    }

    if (UTVehicle(Bot.Pawn) == none)
    {    
        /*
           set bot acceleration steering towards target.
           bots use acceleration steering just like players do.
           noticed that standard ut game bots cheat by directly altering their velocity sometimes.
           however, bots should only use acceleration steering, having the same limits as players.
           nor do we mess with the base pawn AccelRate, because players can't. A player can
           fire their AccelRate in any horizontal directional degree, or not fire it. So in below
           algorithm bots do the same. 
                 
           the steering algorithm creates an acceleration vector which is between the target and the reverse of
           velocity. this is the vector required to keep steering bot towards target yet also damping any 
           velocity pointing away from target.
           
           take for example a bot which needs to move straight ahead to goal and yet is drifting sideways to it's right.
           the needed vector is 45 degrees to it's left, which is the vector in between the reverse of velocity
           (straight left) and the target(staight ahead). this would damp the right hand drift and keep moving bot
           straight ahead. we would not just want to go straight left (to damp the right drift) because then bot's no
           longer going forward to goal.
           
           the steering algorithm automatically limits the bot's speed as needed. if the velocity is not pointing
           at target then bot will accelerate or deccelrate accordingly.
           
           it's important bot doesn't go too fast as they near target as they could hit a wall or overshoot. 
           if air control is limited, such as from jump pad, then it's harder to slow down, so extra speed dampening
           is used.
        */

        //if true we use acceleration, otherwise just drift
        accelerate = true;
        
        //don't want to steer when far away, as we have plenty of room for course
        //adjustments, and trying to steer would limit speed, but when further away
        //you need to go fast as possible
        if ((hDist <= 1000) && (Bot.Pawn.Physics == PHYS_Falling))
        {
            if ((WorldInfo.TimeSeconds - SteerTime) > SteerDuration)
            {
                Steer = ! Steer;
                SteerTime = WorldInfo.TimeSeconds;                    
                
                if (Steer)
                {
                    SteerDuration = UTBRGame(WorldInfo.Game).SteerOnDuration;
                }
                else
                {
                    SteerDuration = UTBRGame(WorldInfo.Game).SteerOffDuration;
                    
                    /*  
                    not used at moment
                    if ((zDir != -1) && (UTBRGoal(MoveTarget) != none))
                    {
                       //allow less drift when jumping up to goal or bot can easily overshoot and hit top rim
                       SteerDuration *= (2/3);                    
                    } 
                    */                   
                }
            }

            //when real close need to steer hard or bot will drift too much, not get to target 
            //quickly and might miss.                                              
            if (hDist <= 150)
            {
                Steer = true;
            }
            
            if (
                (zDir != -1) && (UTBRGoal(MoveTarget) != none) && 
                ((hDist <= 300) || (abs(zdist) <= 150))
                )
            {
                //when going up to goal need more steering close to target
                //or bot can easily overshoot and hit top rim.
                Steer = true;
            }            
            
            if (! Steer)
            {
                //continuously applying steering makes bot shake a lot.
                //continuously applying acceleration makes bot tend to overshoot or undershoot target.
                //a combination of steering and acceleration is better but makes bot flight look choppy.
                //however, turning off acceleration for short periods of time and then switching back to
                //steering makes a nice smooth and accurate flight.
                accelerate = false;
            }                 
        }
        else
        {
            Steer = false;
        }
        
        if ((Bot.Pawn.Physics == PHYS_Falling) && Blocked)
        {
            //need tight navigation when clearing blockages
            Steer = true;
        }        
                   
             
        if ((UTBRGoal(targetActor) == none) && Steer && (Bot.Pawn.AirControl < Bot.Pawn.Default.AirControl))
        { 
            //check if bot needs to slow down. currently only needed if aircontrol is
            //limited, as then it's harder to slow down so we need the extra slow down
            //given here. not needed for goals as you actually want to go at max speed
            //into the goal and generally don't need to worry about overshooting it, as it's
            //not a tight nav point you're trying to land on, but a wide mouth you're trying
            //to hit.
            
            //only slow down if velocity is pointing approximately towards target.
            //this will slow bot down yet bot keeps moving towards target.
            //if velocity is not pointing at target, then just let steering vector take care of it,
            //otherwise the bot would slow down but would not be steered towards target,
            temp = normal(Bot.Pawn.Velocity);
            temp.Z = 0;
            temp = normal(temp);
            temp = Bot.Pawn.Location + (temp * hDist);
            if (HorizontalDistanceBetweenPoints(temp, target) < 200)
            {        
                //get horizontal velocity, the speed they are moving 
                //horizontally. we don't want to confuse this with their
                //regular velocity as that includes speed they are moving
                //vertically also.
                temp = Bot.Pawn.Velocity;
                temp.Z = 0;
                hVel = vsize(temp);
                velAdj = 50;

                //make them slow down more as they get nearer target
                if (hVel > (hDist + velAdj))
                {
                    slowDown = true;
                }
            }
        }
      
        if (Steer)
        {
            //the steering vector which keeps bot moving towards target while at same time
            //dampening any velocity not pointing at target
            temp = (Normal(Bot.Pawn.Velocity) * -1) + Normal(target - Bot.Pawn.Location);
            
            if (vsize(temp) == 0)
            {
                //this happens when velocity is pointing perfectly at target. a zero vector results,
                //so just set vector to point at target.
                temp = target - Bot.Pawn.Location;
            }                         
        }
        else
        {
            //accelerate straight at target
            temp = target - Bot.Pawn.Location;         
        }      

        if (slowDown)
        {
            //bot going too fast. accelerate in reverse of velocity to slow down.        
            temp = Bot.Pawn.Velocity * -1;
        }                       

        //see PlayerController. Acceleration is always horizontal for jumping or walking.
        //for swimming you can accelerate in all directions.
        if (Bot.Pawn.Physics != PHYS_Swimming)
        {                    
            temp.z = 0;
        }

        //take the Normal after cropping Z, not before, otherwise you would truncate the vector length.        
        temp = Normal(temp);

        if (accelerate)
        {
            Bot.Pawn.Acceleration = temp * Bot.Pawn.AccelRate;        
        }
        else
        {
            Bot.Pawn.Acceleration = vect(0,0,0);        
        }
    }
    
    if (UTVehicle(Bot.Pawn) != none)
    {
        /*
        we use special navigation for vehicles too because regular ut bots don't do so well, always getting stuck.

        for vehicles we use FindAir state which allows us to use ut navigation to steer to a vector. 
        for other navigation states we use MoveToGoal which doesn't use ut navigation at all and we 
        control navigation ourselves by uses Acceleration.
        Acceleration does not work on vehicles. to navigate a vehicles ourselves we would have to use vehicle.SetInputs.
        it's generally better to use ut navigation to steer the vehicle to a vector 
        because steering vehicles is complicated. you just can't move a vehicle in a straight line to a place,
        it could involve using front/back and left/right in a series of movements.
        in spite of this, sometimes we use SetInputs, but only when needed.
        we don't need as tight control over vehicles as we do for jumping, as
        vehicles don't have all the drift issues that jumping has.        
        Using SetInputs however has some technical issues because always fighting with
        ut over them so our throttle settings get overriden by ut and don't get used.
        if we limit out throttle usage to just a few limited situations, it seems fine.
        */

        /*                    
        //some possible code for SetInputs if we ever could figure out how to stop ut from overridding our settings
        Bot.GotoState('MoveToGoalWithEnemy');                                   
        GetAxes(Bot.Pawn.Rotation,X,Y,Z);
        InForward = DistanceBetweenPoints(Bot.Pawn.Location + (X * -10), Destination) - DistanceBetweenPoints(Bot.Pawn.Location + (X * 10), Destination);
        InStrafe = DistanceBetweenPoints(Bot.Pawn.Location + (Y * -10), Destination) - DistanceBetweenPoints(Bot.Pawn.Location + (Y * 10), Destination);
        InUp = DistanceBetweenPoints(Bot.Pawn.Location + (Z * -10), Destination) - DistanceBetweenPoints(Bot.Pawn.Location + (Z * 10), Destination);
        InForward = InForward / abs(InForward);
        InStrafe = InStrafe / abs(InStrafe);
        InUp = InUp / abs(InUp);             
        UTVehicle(Bot.Pawn).SetInputs(Inforward, InStrafe, InUp);
        */           
            
        
        //using FindAir because it calls MoveTo(Destination) without doing much else
        //can't call MoveTo(Destination) except thru a bot State
        //must reset state continuously or bot may stop navigating and just float around
                   
        Bot.GotoState('FindAir', 'DoMove');
        LastDestination = Destination;
        
        
        if (BackingUp)
        {
            /*
               it's sad to have to override the throttle directly like this, but
               if we use ut3 navigation the bot often will not backup and get stuck
               because it's trying something stupid like a frontal turn against the
               blockage trying to reach the point directly behind it.
               strangely, in this situation, calling SetInputs seems to be fine, as ut
               is not overriding us (don't know why)
            */
            Bot.GotoState('MoveToGoalWithEnemy');  //must use this state or ut will override our SetInputs settings
            
            //use a random strafe to give the vehicle a little wiggle so it's not just a straight backup,
            //as there might be something behind vehicle, and to point the nose of vehicle away from blockage
            //a little so bot has better change of clearing it when it starts going forward again
            UTVehicle(Bot.Pawn).SetInputs(-1, rand(3) - 1, 0);
        }
        
    }
   
    SetUTBotMoveTarget(); 
}

//bot starts a fresh path to objective after trying other things
function RestartPathCycle()
{
    TurnOffBlocking();
    SetPathSubTarget(none); 
    TriedDefensivePosition = false;
    SpecialNavigationPathTarget = none;
    Bot.DefensivePosition = none;
    ForceRepath = true;
    SoakTarget = none;
}

function bool CheckSpecialNavigationMoveTargetChange()
{
    if ((MoveTarget != SpecialNavigationPathTarget) && (SpecialNavigationPathTarget != none) &&
        (MoveTarget != none))
    {
        ShowPath("Ending forcing of special navigation. MoveTarget changed from " $ SpecialNavigationPathTarget $ " to " $ MoveTarget);               

        SpecialNavigationPathTarget = none;
        JumpStartedByPlayerManager = false;
        FindingRandomTarget = false;        
        
        return true;
    }    
    
    return false;
}

/*
   turn on the special navigation which helps bot navigate.
   uses special steering to navigate bot accurately through air when jumping(regular ut
   bot jumps are very inaccurate).
   makes sure a bot takes advantage of quad jump (regular ut does not).
   when bot encounters a blockage, it will navigate around it.
   has spontaneous target rerouting, so if situation changes while bot is in transit, bot
   will reroute to new target (in regular ut, bot waits till it reaches a path node or lands to retask).
*/
function StartSpecialNavigation(Actor target)
{   
    local string s;
               
    SetMoveTarget(target);     
    
    SpecialNavigationPhysics = Bot.Pawn.Physics;
           
    if (Bot.Pawn.Physics == PHYS_Falling)
    {
        //pawn is jumping or falling
        
        if (UTBRGame(WorldInfo.Game).Test_NoJump)
        {
            return;
        }              
           
        LastJumpStartLocation = vect(0,0,0);
    
        JumpStartTarget = MoveTarget;
        
        JumpStartLocation = Bot.Pawn.Location;
                
        KeepJumping = true;      
                       
        Steer = false;
        SteerTime = WorldInfo.TimeSeconds;
        SteerDuration = 0;
    }
      
    LastZ = Bot.Pawn.Location.Z;        
    SpecialNavigating = true;
    LastDestination = vect(0,0,0); 
    
    SoakSpecialNavigationTarget = target;
    SpecialNavigationFormationCenter = FormationCenter;
    LastFormationCenterLocation = FormationCenterLocation;
    ClosestDistanceToFormationCenter = DistanceFromFormationCenter;
    Reason = "";
    ForceRepath = false;

    SetBotTacticTimer();
    
    s = string(SpecialNavigationPhysics);

    if (Bot.Pawn.Physics == PHYS_Falling)
    {
        s = "Jumping";
    }
        
    if (Bot.Pawn.Physics == PHYS_RigidBody)
    {
        s = "Driving";
    }
    
    if (TurningForGoalJumpShot)
    {
        s = "Turning for goal jump shot";
    }
    
    if (FindingRandomTarget)
    {
        s = s $ ", Finding random target";
    }    
        
    ShowPath("Start special navigation: " $ s $ "  MoveTarget=" $ MoveTarget $ " PathTarget=" $ PathTarget $ " SpecialNavigationPathTarget=" $ SpecialNavigationPathTarget);
}

//return true if doing a jump for which we measure the results upon landing to
//see if another jump is good or not
function bool PerformingMeasuredJump()
{
    return (SpecialNavigationPathTarget != none) && JumpStartedByPlayerManager && (! FindingRandomTarget);
}

//return true if bot is trying to reach target using forced special navigation.
//bot will continue to use special navigation until it reaches target.
//used to resolve pathing difficulties.
function bool ForcedSpecialNavigating(optional Actor target)
{
    if (target == none)
    {
        return SpecialNavigating && (SpecialNavigationPathTarget != none);
    }
    
    return SpecialNavigating && (SpecialNavigationPathTarget != none) && (SpecialNavigationPathTarget == target);    
}

//end navigation being performed by SpecialNavigate().
//force: force to end because of failure. current path is aborted.
function EndSpecialNavigation(optional bool forceFailFromSoak)
{
    local float diff, diff2, minDist, minDist2;
    local bool jumpZOk, allowRejump, navFailed, pendingNav, wasMeasuredJump;
    local Vector temp;
    local string s;
    
    if (! SpecialNavigating)
    {
        return;
    }
    
    
    if (Reason != "")
    {
        Reason = ": " $ Reason;
    }
    
    s = "";
    if (forceFailFromSoak)
    {
        s = "(force fail)";
    }

    navFailed = false;
    if (forceFailFromSoak)
    {
        navFailed = true;
    }

    CheckSpecialNavigationMoveTargetChange();
            
    ShowPath("End special navigation" $ s $ " " $ Reason $ "  MoveTarget=" $ MoveTarget $ "(reached " $ ReachedTarget(MoveTarget) $ ") PathTarget=" $ PathTarget $ " SpecialNavigationPathTarget=" $ SpecialNavigationPathTarget);
   
    ReachedTarget(MoveTarget);  //make sure route history is updated
    
    pendingNav = 
     ForcedSpecialNavigating() && (! navFailed) &&
     (KeepMovingWhenReached() || (! ReachedTarget(MoveTarget)));
     
              
    if (SpecialNavigationPhysics == PHYS_Falling)
    {
        LastJumpTime = WorldInfo.TimeSeconds;
    }
      
    if (PerformingMeasuredJump()) 
    {              
        wasMeasuredJump = true;
        
        //soak isn't checked during this jump because we measure the success of the jump
        //upon landing. now that landed, reset soak timer so soak checks can resume from
        //this point on
        SoakTime = WorldInfo.TimeSeconds;
        
        diff = vsize(JumpStartLocation - Bot.Pawn.Location);
        minDist = 25;

        diff2 = vsize(LastJumpStartLocation - Bot.Pawn.Location);
        minDist2 = vsize(LastJumpStartLocation - JumpStartLocation);
        
        jumpZOk = (MoveTarget != none) && CanJumpToHeightOf(MoveTarget);            
  
        
        allowRejump = false;
        
        if ((diff > minDist) && jumpZOk)
        {
            //bot has moved and is able to jump up to target, so try again
            allowRejump = true;
        }            
        
        if (
            (abs(LastJumpStartLocation.Z - JumpStartLocation.Z) > 175) &&
            (abs(Bot.Pawn.Location.Z - JumpStartLocation.Z) < 50)
           )
        {
            //Z has changed enough to keep jumping
            LastJumpStartLocation = vect(0,0,0);            
        }
        
        if ((LastJumpStartLocation != vect(0,0,0)) && (diff2 <= minDist2))
        {
            //bot may have moved, but if just jumping back and forth then stop
            allowRejump = false;
        }
        
        if (JumpingOverBlockage)
        {
            //if bot was only trying to clear a short blockage then allow rejump, as this is not related to path failure
            allowRejump = true;
        }
        
        if (! allowRejump)
        {
            navFailed = true;
        }
        
        if (ReachedTarget(MoveTarget))
        {
            allowRejump = false;
            navFailed = false;
            pendingNav = false;            
        }
        
        if (ReachedTarget(MoveTarget) && CanInitiateJump() && 
            (UTBRGoal(MoveTarget) != none) && (GetBall(Bot.Pawn) != none))
        {
            //must be some problem hitting goal so keep hammering away till scores
            allowRejump = true;
            ForceJump = true;
            navFailed = false;
        }
               
        if ((GetBall(MoveTarget) != none) && (GetBall(Bot.Pawn) != none))
        {
            allowRejump = false;
            navFailed = false;
            pendingNav = false;            
        }          
                  
        //after failed jump near to target fall back to another position so that bot doesn't just loop back and forth.
        //if jump improved bot position then keep going.
        if (allowRejump)
        {
            ShowPath("Allow rejump. Dist diff=" $ diff $ ", diff2=" $ diff2 $ ", minDist2=" $ minDist2 $ ", blocked=" $ Blocked);
        }
        
        if (navFailed)
        {
            pendingNav = false;
            ShowPath("Jump failed. diff=" $ diff $ " jumpZOk=" $ jumpZOk $ " LastJumpStartLocation=" $ LastJumpStartLocation $ " diff2=" $ diff2 $ " minDist2=" $ minDist2);
        }
                
        //ball was shot during jump and bounced back to bot.
        //make sure variables don't get reset due to the temporary formation center change. 
        if (
            GoalJumpingWithBall && 
            (GetBall(Bot.Pawn) != none) && 
            ((UTBRBall(LastFormationCenter) != none) || (LastFormationCenter == none))
           )
        {           
            LastFormationCenter = FormationCenter;
        }       
    }
   
    if (FindingRandomTarget)
    {
        //reset soaking vars for a fresh soak test
        SoakTarget = none;
    }
   
    if ((SpecialNavigationPhysics == PHYS_Falling) && (SpecialNavigationPathTarget == none) && (! FindingRandomTarget))
    {
        //some kind of jump occured which wasn't a forced special navigation
        
        if (IsFormationCenter(MoveTarget))
        {
            //bot temporarily deviated from route to jumped at objective, now continue route.
            //examples of this are forward dodging or random jump to objective.
        }     
        else 
        {
            //a jump initiated by ut was being processed.
            
            //used to stop repathing from being done, but too many situations
            //came up where repathing was needed, such as falling off a ledge.
            //potential problem with repathing is that ut will make bot go back
            //to a node already reached. however, FindPathToObjective will detect
            //this and force bot onward to another node.
        }
    }
   
    if (navFailed)
    { 
        SoakTarget = none;
        SuppressDefensivePositionReset();
        CheckPathSubTargetReset();
        
        if (! ForcedSpecialNavigating(PathTarget))
        {
            FailPath();
        }
        else if (IsFormationCenter(PathTarget))
        {
            if (TriedDefensivePosition)
            {
                /*
                bot had been going towards defensive position and then had
                routed to formation center when navigation failed.
                we have tried just about everything now so seek a random place
                and start the cycle over.
            
                FindingRandomTarget is the ultimate last resort when in bad pathing situation.
                Bot can be in a real tight spot where absolutely all pathing fails to any node
                whatsoever, and special navigation also is failing.
                By going to a random point directly without using pathing, bot can get out of that
                place where hopefully pathing or special navigation will start working again.
                */
                
                if (forceFailFromSoak || IsSoaking)
                {
                    //only want to find random target if was soaking to get it
                    //out of bad spot, otherwise finding random target can put
                    //bot in bad spot
                    
                    ShowPath("Special navigation failed. Finding random target");                
                    FindingRandomTarget = true;
                    TurnOffBlocking();               
                    pendingNav = true;
                    TriedDefensivePosition = false;
                    SpecialNavigationTime = WorldInfo.TimeSeconds;
                    FindRandomDestination();
                }
                else
                {
                    ShowPath("Special navigation failed. Restarting path cycle.");
                    RestartPathCycle();
                }               
            }
            else
            {
                //pathing to formation center had failed so bot was using special 
                //navigation which has now failed. go to DefensivePosition in hopes
                //navigation will work better from there.
                
                ForceDefensivePosition = true;
                ShowPath("Special navigation failed. Forcing defensive position.");
            }
        }
        else if (PathTarget == Bot.DefensivePosition)
        {
            //this should not happen a lot if at all, but here as a safety check.
            //don't want to use special navigation to reach defensive position since
            //from there we special navigate to formation center, so special navigating
            //to defensive position would just waste time, we should rather just go to
            //formation center
            
            DefensivePositionFailed = true;
            Bot.DefensivePosition = none;
                               
            if (SeekDirect)
            {
                ShowPath("Special navigation to DefensivePosition failed. Routing special navigation to " $ FormationCenterTarget);
                SetMoveTarget(FormationCenterTarget);
                pendingNav = true;
            }
            else
            {
                ShowPath("Special navigation to DefensivePosition failed.");       
            }
        }
        else if (GettingSpecialItem())
        {
            ShowPath("Failing special navigation to special item " $ PathTarget);
            
            //this makes path to be given up on. bot will stop seeking this item
            //for a time. special item search functions will ignore this item so
            //another can be selected.                
            FailedTarget = PathTarget;
        }
    }

    InitSpecialNavigation();   

    if ((SpecialNavigationPhysics == PHYS_Falling) && (Bot.Pawn.Physics != PHYS_FALLING))
    {
        JumpingOverBlockage = false;
    }     
    
    //now see if special navigation needs to be restarted
                                
    if (ForceJump)
    {
        ForceJump = false;
        temp = JumpStartLocation;

        SpecialNavigationPathTarget = MoveTarget;
             
        if (TrySpecialJump(MoveTarget))
        {
            if (wasMeasuredJump)
            {
                LastJumpStartLocation = temp;
            }                
        }
    }
        
    if (Bot.Pawn.Physics != PHYS_FALLING)
    {
        JumpingOverBlockage = false;
    }         
       
    if ((! SpecialNavigating) && pendingNav)
    {
        ShowPath("Restarting special navigation");
        TrySpecialNavigation(MoveTarget, true);
    } 

    if (! SpecialNavigating)
    {
        InitSpecialNavigation(true);
    
        ForceRepath = true;
        
        SetUTBotMoveTarget();
        
        //need this to keep ut going to target until retask occurs        
        Bot.SetAttractionState();
        
        Retask();
    }
}

//see if PathSubTarget needs to be reset due to navigation failure
function CheckPathSubTargetReset()
{
    if ((PathSubTarget != none) && (MoveTarget == PathSubTarget))
    {
        ShowPath("Resetting path sub target " $ PathSubTarget $ " because navigation failed");
        SetPathSubTarget(none);
    }
}

function SuppressDefensivePositionReset()
{
    local float newValue;
    
    if (IsFormationCenter(PathTarget) || (PathTarget == Bot.DefensivePosition))
    {
        //if navigation isn't working so well, then force bot to defensive position until it gets
        //vertically closer to target
        newValue = abs(VerticalDistanceFromFormationCenter) - 25;
        
        if (newValue < SuppressDefensivePositionResetZ)
        {
            SuppressDefensivePositionResetZ = newValue;
        }
        else
        {
            SuppressDefensivePositionResetZ -= 25; 
        } 
        ShowPath("Supressing defensive position reseting at current Z");
    } 
}

//find a good place to do goal jump, somewhere in front of goal
//try to place bot close enough that jump can be made, but if 
//jump can't be made then put bot far enough away to do a jump shot.
function FindGoodGoalJumpPosition(UTBRGoal goal)
{
    GoalJumpPosition = MoveAwayFromGoalFaceVector(Bot.Pawn, GoalJumpDist(goal), goal);      
}

//return dist away from goal bot should be to jump
function float GoalJumpDist(UTBRGoal goal)
{
    local float dist, zDist;
  
    zDist = goal.Location.Z - Bot.Pawn.Location.Z;
    
    //good horizontal dist away from goal to start jump     
    dist = JumpApexHorizontalDistance;
    
    if (HorizontalDistanceBetween(Bot.Pawn, goal) < dist)
    {
        //bot already close enough so keep it at that distance (but it may still need to get in front of goal)
        dist = HorizontalDistanceBetween(Bot.Pawn, goal);    
    }
    
    if (dist < 50)
    {
        dist = 50;
    }

    if ((! CanJumpToHeightOf(goal)) &&
        (zDist > (JumpHeight + 150))) 
    {
        //bot can't make jump so put bot out at a good distance for jumping shot
        //however if jump is close enough then don't need to. 
        dist = zDist;
    }
    
    if (dist > 600)
    {
        dist = 600;
    }
    
    return dist;
}

//return true if bot can jump to height of target
function bool CanJumpToHeightOf(Actor target, optional float startingZ)
{
    local float variance;
    local float zDist;
    local vector loc;
    
    local float dist;
    
    variance = 0;
    loc = target.Location;
    
    if (UTBRGoal(target) != none)
    {
        //find out how much clearance needed to hit goal
        
        variance = 0;
        
        if ((abs(target.Rotation.Roll) <= 8000) || (abs(target.Rotation.Roll) >= 24000))
        {
            //vertical goal
            
            variance = 100;
            
            //trace down to goal rim
            dist = BRGame.BlockedDist(target, vect(0,0,-1), 250, target.location, vect(50,50,50));
            if (dist > 0)
            {
                //add extent back in
                dist += 50;
            }
            
            if ((dist > 0) && (dist <= 250))
            {
               //subtract out approx dist from waist to feet so bot can clear rim
               variance = dist - 50;
               if (variance < 0)
               {
                   variance = 0;
               }
            }
        }
    }
    
    if (UTBRBall(target) != none)
    {
        loc = UTBRBall(target).Position().Location;
    }

    if (startingZ == 0)
    {    
        startingZ = Bot.Pawn.Location.Z;
    }
    
    zDist = loc.Z - startingZ;
            
    return zDist <= (JumpHeight + variance);
}

//return true if bot can jump to target
function bool CanJumpTo(Actor target, optional vector startLoc)
{    
    if (startLoc == vect(0,0,0))
    {
        startLoc = Bot.Pawn.Location;
    }

    return
        CanJumpToHeightOf(target, startLoc.Z) &&
        CloseEnoughToJump(target);
}

function bool CloseEnoughToJump(Actor target, optional vector startLoc)
{
    if (target == none)
    {
        return false;
    }
    
    if (startLoc == vect(0,0,0))
    {
        startLoc = Bot.Pawn.Location;
    }
    
    return (HorizontalDistanceBetweenPoints(startLoc, target.Location) <= (JumpApexHorizontalDistance + 25));
}

//makes a bot try to shoot ball into goal.
//sets up a timer which makes bot continuously try for a few seconds to get a lined up and clear shot.
function TryShootingGoal(UTBRGoal goal)
{  
    if (ShootingGoal || UTBRGame(WorldInfo.Game).Test_NoShoot)
    {
        return;
    }
    
    BotShootTime = WorldInfo.TimeSeconds;
    GoalTarget = goal;
    SetFocalPoint(GoalTarget.Location);
    ShootingGoal = true;
    SetBotTacticTimer();
}

function ShootBallAtGoal(UTBRGoal goal)
{
    if (UTBRGame(WorldInfo.Game).Test_NoShoot)
    {
        return;
    }
    
    ShowPath("Shoot ball at goal");
    
    GoalTarget = goal;
    
    ShootBall();                  

    Bot.Focus = TheBall;
    
    if (HasBoostWeapon(true) && ShouldBoostBall())
    {
        AttackBallBoosting = true;
        ShowPath("Attack ball boosting");
    }
}

//return true if bot should boost ball, based on skill level
function bool ShouldBoostBall()
{
    local int i;

    i = 3;
    
    if (HorizontalDistanceBetween(Bot.Pawn, Squad.FriendlyGoal) <= 3000)
    {
        i = 2;
    }
    
    if (Bot.Skill >= 5)
    {
        i = 2;
    }
      
    return rand(i) == 0;    
}

//return true if bot is over target
function bool OverTarget(Vector target)
{
   local float hortDist, vertDist, dist;
     
   if (target == vect(0,0,0))
   {
       return false;
   }
     
   vertDist = Bot.Pawn.Location.Z - target.Z;
   hortDist = HorizontalDistanceBetweenPoints(Bot.Pawn.Location, target);


   if (vertDist > 0)
   {
       if (hortDist <= 100)
       {       
           return true;
       }
                
       dist = hortDist * 0.50;
       
       if ((Bot.Pawn.Physics == PHYS_Falling) &&        
          ((target.Z - JumpStartLocation.Z) >= 300))
       {
           //give a little more clearance for targets in air to make sure bot makes it
           dist = hortDist * 0.60;           
       }
                
       if (vertDist > dist)
       {     
           return true;
       }      
   }
     
   return false;
}

//called by CheckForZSurface
function bool CheckForZSurface2(vector dir, float dist, float depth, optional vector extent)
{
    local vector hitLocation, hitNormal, startTrace, endTrace;
    local Actor actor;
    
    startTrace = Controller.Pawn.Location + (dir * dist);
            
    endTrace = startTrace;
    endTrace.Z = endTrace.Z - depth;
    
    foreach Controller.Pawn.TraceActors( class'Actor', actor, hitLocation, hitNormal, endTrace, startTrace, extent,,)
    {
        if (actor.bBlockActors)
        {     
            return true;
        }
    }    
    
    return false;
}

//return distance floor or ceiling is found checking maxDist in dir at depth.
//positive depth for floor, negative for ceiling.
function float CheckForZSurface(vector dir, float maxDist, float depth, optional vector extent)
{
   local float dist, goodDist;
   
   dir = Normal(dir);

   ZSurfaceBreakDist = -1;
   
   while (dist < maxDist)   
   {
       dist = dist + 25;
       if (dist > maxDist)
       {
           dist = maxDist;
       }
       
       if (! CheckForZSurface2(dir, dist, depth, extent))
       {
          //break in surface found
          ZSurfaceBreakDist = dist;
          break;
       }
    
       goodDist = dist;
   }

   ZSurfaceDist = goodDist;

   return goodDist;
}

//return distance of floor in direction bot is facing 
function float CheckForFloorFacing()
{
    return CheckForZSurface(vector(Bot.Pawn.Rotation), 2000, 200.0);
}

//return distance of floor between bot and loc
function float CheckForFloorUpTo(vector loc, optional float dist)
{
    loc.z = Bot.Pawn.Location.Z;
    
    if (dist == 0)
    {
        dist = HorizontalDistanceBetweenPoints(loc, Bot.Pawn.Location);
    }

    return CheckForZSurface(loc - Bot.Pawn.Location, dist, 100);
}

//return true if there is floor between bot and loc
function bool GoodFloorUpTo(vector loc, optional float dist)
{
    if (dist == 0)
    {
        dist = HorizontalDistanceBetweenPoints(loc, Bot.Pawn.Location);    
    }
    
    return CheckForFloorUpTo(loc, dist) == dist;
}

//activate a weapon good for boosting. 
//return true if found one.
function bool HasBoostWeapon(bool setActive)
{
    local UTWeapon w;
    local Pawn P; 
   
    if ((Bot == none) || (Bot.Pawn == none))
    {
        return false;
    }
    
    P = Bot.Pawn;
    if (Vehicle(P) != none) 
    {
        if (UTVehicle_Hoverboard(P) != none)
        {
            P = Vehicle(P).Driver;
        }
        else
        {
            return false;
        }
    }   
    
    ForEach P.InvManager.InventoryActors( class'UTWeapon', w )
    {
        if (IsBoostWeapon(w))
        {
             if (setActive)
             {
                 LeaveVehicle();
                 P.SetActiveWeapon(w);
             }
             return true; 
        }
    }
    
    return false;
}

//return true if weapon is good for boosting (shooting object to knock it a distance)
function bool IsBoostWeapon(Weapon w)
{
    return 
    w.HasAnyAmmo() && 
    (
     (UTWeap_InstagibRifle(w) != none) || (UTWeap_ShockRifle(w) != none) || 
     ((InStr(CAPS(w.name), "INSTAGIB") >= 0) && (InStr(CAPS(w.name), "RIFLE") >= 0))
    );
}


//return true if C is driving a wheeled vehicle
function bool DrivingWheels()
{
    return (UTVehicle(Controller.Pawn) != none) && (UTVehicle(Controller.Pawn).Wheels.Length > 3) 
           && (UTHoverWheel(UTVehicle(Controller.Pawn).Wheels[3]) == none);
}

//called by SuggestStrafeDestination to rate a possible place to strafe to
function EvaluateStrafeLocation(
    Vector loc, 
    bool isZLoc,
    bool bestIsZLoc,
    out Vector best, out float bestDist, out float bestClearance, out float bestDistFromBot
    )
{
    local float dist, dist2, traceDist, distFromBot, clearance, traceDist2, len;
    local Vector dir, traceLoc, dir2;
    local bool ok;
    local Controller C;
    local Vector extent;

    //useful for debuging
    //spawn(class'UTEmit_VehicleHit',,,loc);
     
    C = Controller;
    
    dist = HorizontalDistanceBetweenPoints(Destination, loc);
    
    dist2 = DistanceBetweenPoints(Destination, loc);             

    extent = GetSearchTraceExtent();
    
    dir = Normal(Destination - loc);                
    traceDist = BRGame.BlockedDist(self, dir, dist2, loc, extent);
    if (traceDist == 0)
    {
        traceDist = dist2;
    }    
    
    //important to trace out horizontally also and use best dist, as target could
    //be above/below loc and first trace can run into ceiling/floor
    dir2 = dir;
    dir2.Z = 0;
    dir2 = normal(dir2);
    traceDist2 = BRGame.BlockedDist(self, dir2, dist2, loc, extent);
    if (traceDist2 == 0)
    {
        traceDist2 = dist2;
    }
    
    if (traceDist2 > traceDist)
    {
        traceDist = traceDist2;
        dir = dir2;    
    }              
    
    traceLoc = loc + (dir * traceDist);       
    
    distFromBot = HorizontalDistanceBetweenPoints(C.Pawn.Location, loc);
    
    //calc how far pawn can travel towards target by going to traceLoc
    clearance = (traceLoc - C.Pawn.Location) dot normal(Destination - C.Pawn.Location);
     
    ok = false;
    if (clearance >= (bestClearance - 25))
    {
        ok = true;
    }
    
    if ((! ZBlocked) && (! IsZloc) && FacingOppositeDirection(loc))
    {
        //for navigating horizontally, we only want to consider places that would make bot strafe. 
        //we don't want to back up.
        ok = false;
    }    
      
    if (IsZLoc && ZBlocked)
    {
        ok = false;
        
        //allow for fluctuations in terrain for clearing vertical blockages
        if (clearance >= (bestClearance * (3/5)))
        {
            ok = true;
        }        
    }
        
    if (! ok)
    {
        return;
    }
    
    if (! IsZLoc)
    {
        //if blocked vertically, zLoc is better
        if (ZBlocked && bestIsZLoc)
        { 
            ok = false;
        }

        //for horizontal paths, choose the one which is closest to target                    
        if (dist >= bestDist)
        {             
            ok = false;
        }
        
    }
    
    //for vertical paths, choose the one which is closest to bot.
    //very important because this will tend to make bot clear the
    //veritcal blockage by moving to the cloest clearance point and
    //thus clear the blockage quickly, otherwise bot might
    //rejump and hit the blockage (if blockage if above bot)
    if ((isZLoc) && (distFromBot >= bestDistFromBot))
    {             
        ok = false;
    }        

    if (DistanceBetweenPoints(traceLoc, loc) <= 25)
    {
        //no use going to this trace if it's stuck too
        ok = false;
    }

    if ((! isZLoc) && BRGame.BlockedDist(self, BlockedDirection, BlockedDist) > 0)
    {
        //reject a location if it just takes bot back into blockage.         
        //however, if a zero extent trace doesn't block then vehile is not blocked right in front but
        //to the side, so we can then allow the strafe loc to be more in front
        //not doing the zero extent trace can result in bot going back and forth because as it turns
        //away from a wall towards the strafe loc the bot is facing that loc more and if bot gets blocked again
        //by the same wall some other loc too far out to the side would get choosen
            
        len = 0.8;
        
        if (DrivingWheels())
        {
            //wheeled vehicles need to turn more to the side to clear a blockage
            //otherwise they get stuck driving head on into blockage   
            len = 0.6;
        }
        
        if (FacingGeneralDirection(loc, len))
        {        
            ok = false;
        }
    }
    
    if (! ZBlocked)
    {
        //for horizontal blockage, to use a zloc instead of a horizontal loc, 
        //the clearance must be much better
            
        if (isZLoc && (! bestIsZLoc) && ((clearance - bestClearance) < 100))
        {
            ok = false;
        }
        
        if ((! isZLoc) && bestIsZLoc && ((clearance - bestClearance) > -100))
        {
            ok = true;
        }        
    } 

    
    if (! ok)
    {
        return;
    }

    
    //found best
        
    bestDist = dist;
    bestClearance = clearance;
    
    if (isZLoc)
    {
        bestDistFromBot = distFromBot;
    }
    
    best = loc;              
}

//check if bot blocked vertically from reaching it's target
function CheckForVerticalBlockage()
{
    local float distCheck, zDist, hDist;
    local Vector zDir, vTemp;
    local Controller C;
    local float cRadius, cHeight;
    local Actor blockedBy;

    C = Controller;
    
    hDist = HorizontalDistanceBetweenPoints(C.Pawn.Location, Destination);
    
    zDist = abs(Destination.Z - C.Pawn.Location.Z);   
    zDir = vect(0,0,0);    
    zDir.Z = ZDirMult;
    
    C.Pawn.GetBoundingCylinder(cRadius, cHeight);        
        
    //for vertical blockages bot needs more dist to check because more leeway is needed due to gravity and rejumps
    //drastically altering bot's vertical position 
    distCheck = zDist;
    if (distCheck > 200)
    {
        distCheck = 160 + cHeight;
    }
    
    if ((UTVehicle(C.Pawn) != none) && (UTVehicle(C.Pawn).bCanFly)) 
    {
        //flying vehicles hover over surface so need more distance for the check
        distCheck += 200;
    }        

    zBlockDist = BRGame.BlockedDist(self, zDir, zDist, C.Pawn.Location, GetSearchTraceExtent());
    blockedBy = BRGame.BlockingActor; 
    
    BlockedZ = C.Pawn.Location.Z + (zBlockDist * ZDirMult);
       
    if (
        (zBlockDist > 0) && 
        (zBlockDist <= distCheck) && 
        ((zDist - zBlockDist) > 75)
       )
    {              
        vtemp = Destination;
        vtemp.Z = C.Pawn.Location.Z;

        if ((hDist <= 300) && (BRGame.BlockedDist(self, zDir, distCheck, vtemp, GetSearchTraceExtent()) > 0))
        {
            //once bot is directly above or below target and is Z blocked, then
            //and only then set the flag. 
            //this will make the bot move away, even far away, until it clears blockage.
            
            ShowPath("Blocked vertically by " $ blockedBy);                
            
            Blocked = true;
            ZBlocked = true;
        }
    } 
}

//trace out in front and return blocked dist
function float TraceOutFront(float distCheck, float width, vector startLoc)
{
    local vector CrossDir;
    local Controller C;
    local float dist, w, endAt;
    local Vector loc, X, Y, Z;
  
    /*
        we can't just do one trace with a wide trace extent because extents are absolute to the world axis.
        a trace with a wide x/y would end up colliding with the ground under pawn when pawn's on an
        incline. only recourse is to do several small extent traces across front of pawn along pawn's X vector.
    */
    
    C = Controller;
       

    //bot could be turned and attacking another target while moving
    //to destination so can't use pawn's rotation direction for blockage detection
    BlockedDirection = Normal(Destination - C.Pawn.Location);
    BlockedDirection.Z = 0;
    BlockedDirection = normal(BlockedDirection);
    
    if (DrivingWheels())
    {
        //for wheeled vehicle the pawn does not turn with the aim focus, and
        //don't want blockage detected on pawn's side when it's passing a wall,
        //so detect blockage in the way the pawn is pointed
        GetAxes(C.Pawn.Rotation,X,Y,Z);
        BlockedDirection = X;
    }
    
    //the cross product of two vectors is perpendicular to both 
    CrossDir = Normal(BlockedDirection cross vect(0, 0, 1));


    w = (width / 2) - 25;
    if (w < 0)
    {
        w = 0;
    }
    endAt = 0 - w;
              
    while (true)
    {
        loc = startLoc + (CrossDir * w);
        
        dist = BRGame.BlockedDist(self, BlockedDirection, distCheck, loc, vect(25,25,30));
        //SendDebugMessage("dist=" $ dist);

        if (dist != 0)
        {
            //blocked
            BlockingStartLoc = loc;
            return dist;
        }
        
        if (w == endAt)
        {
            break;
        }
        
        w -= 25;
        if (w < endAt)
        {
            w = endAt;
        }
    }    

    //not blocked
     
    BlockedDirection = vect(0,0,0);
    
    return 0;  
}

//check if bot blocked horizontally from reaching it's target
function CheckForHorizontalBlockage()
{
    local Controller C;
    local float hDist, zDist, dist, distCheck;
    local Vector zDir, vTemp, extent;
    local actor blockedBy;
    local float temp1,temp2, vel;
    local int i;  
    local vector sExtent;  
    

    if (ZBlocked)
    {
        Reason = "ZBlocked";
        return;
    }       
    
    C = Controller;
    
    hDist = HorizontalDistanceBetweenPoints(C.Pawn.Location, Destination);
    
    if ((hDist < 25) && (abs(Destination.Z - Bot.Pawn.Location.Z) > 50))
    {
        //if bot's right over or under target don't bother checking for horizontal
        //blockage because it's moving vertically to target
        Reason = "Over target";
        return;
    }
    
    zDist = abs(Destination.Z - C.Pawn.Location.Z);   
    zDir = vect(0,0,0);    
    zDir.Z = ZDirMult;
       
    extent = GetFrontalBlockageExtent();
    sExtent = GetSearchTraceExtent();
       
    //
    //see if horizontally blocked
    //
    distCheck = 125;
    if (UTVehicle(C.Pawn) != none)
    {        
        distCheck = (extent.X * 2) + 250;    

        vel = 900;
        if (UTVehicle_Hoverboard(C.Pawn) != none)
        {
            vel = 700;
        }
                    
        if (vsize(C.Pawn.Velocity) >= vel)
        {
            distCheck += 200;
            
            if (DrivingWheels())
            {
               distCheck += 100;
            }            
        }
                
        if (UTVehicle_Hoverboard(C.Pawn) != none)
        {   
            //hoverboards drift when they turn, need more clearance
            distCheck += 150;               
        }
    }      
    
    if (distCheck > hDist)
    {
        distCheck = hDist;
    }             

    dist = TraceOutFront(distCheck, extent.x * 2, C.Pawn.Location);                
    blockedBy = BRGame.BlockingActor;

    if (dist > 0)
    {
        Blocked = true;
    }
    
    Reason = "";


    if (Blocked)
    {
        //even though pawn can trace to a blocking place in front, it may not actually
        //be blocked, but rather be on an incline or about to get in an incline.
        //another case is a decline. a vehicle can on a real steep hill with nose slanting down
        //and trace a blockage out front but it's just the terrain.
        
        //do a high and low trace and compare distances to see if this is the case.
    
    
        //first we have to find the high point of the blocking actor,
        //so start at pawn's loc and work down
        //note the little Z in the extent so we get a more accurate distance.
        //the x/y extent must be the same as used in TraceOutFront or these traces could miss.
        
        //BlockingStartLoc is the starting trace loc TraceOutFront used to find blockage.
        
        vtemp = BlockingStartLoc;
        temp1 = 0;
        for (i = 1; i <= 5; i++)
        {
            temp1 = BRGame.BlockedDist(self, BlockedDirection, dist + 200, vtemp, vect(25,25,5));
            if (temp1 != 0)
            {
                break;
            }
            vtemp.z = vtemp.z - 5;
        }
    
        //then compare that trace with a trace at a lower place
        vtemp = BlockingStartLoc;
        vtemp.z = vtemp.z - 35;
        temp2 = BRGame.BlockedDist(self, BlockedDirection, dist + 200, vtemp, vect(25,25,5));         
    
        //if the dist difference from the top point to the low point is too big then
        //we are probably just on an incline and not blocked
        //the numbers 35&25 worked out best in testing. there are a lot of diverse situations
        //which other numbers such as 30&15, 25&10 will fail at. 25 is a fairly useless
        //low point because for minor fluctuations in surface, it would not yield any good
        //difference for both traces. 30 was a grey area. 35 yields a nice clear difference but
        //would not want to go much beyond 35.
            
        if ((temp1 != 0) && (temp2 != 0) && ((temp1 - temp2) >= 25))
        {
            //the blockage is most likely just an incline
            Reason = "Incline detected";
            Blocked = false;              
        }
    } 
            
    if (Blocked && (C.Pawn.Physics == PHYS_Falling))
    {  
        vtemp = C.Pawn.Location;
        vtemp.Z = Destination.Z;
        
        if ((BRGame.BlockedDist(self, zDir, zDist, Bot.Pawn.Location, sExtent) == 0) &&        
            (BRGame.BlockedDist(self, BlockedDirection, hDist, vtemp, sExtent) == 0) 
            )
        {
            //if bot can clear blockage by going up/down to target then don't strafe.
            //this is especially important when goal jumping and hitting rim/wall above/below goal.
            //first trace checks if bot can move up to Z of target, second trace checks from that place
            //to target
            
            Reason = "Can clear by going up/down";
            Blocked = false;       
        }    
    }
    
            
    if (Blocked && (C.Pawn.Physics == PHYS_Swimming))
    {  
        vtemp = C.Pawn.Location;
        vtemp.Z = vtemp.Z + 125;
        
        if ((BRGame.BlockedDist(self, vect(0,0,1), 125, C.Pawn.Location, sExtent) == 0) &&
            (BRGame.BlockedDist(self, BlockedDirection, distCheck + 25, vtemp, extent) == 0) 
            )
        {
            //area above bot is not blocked, so may be able to get back on surface by jumping up, so don't count as blocked
            
            Reason = "Area above not blocked for swimming";
            Blocked = false;        
        }    
    }
    
    if (Blocked && CanInitiateJump())
    {  
        vtemp = C.Pawn.Location;
        vtemp.Z = vtemp.Z + 50;
        
        if (BRGame.BlockedDist(self, BlockedDirection, distCheck + 25, vtemp, extent) == 0) 
        {
            //bot is at a step or something like that which it can jump over.
            //important to do this. target could be in a hole with a rim around the hole
            //and only thing to do is jump over rim.
            
            Reason = "Looks like can jump over blockage";
            JumpOverBlockage = true;
            Blocked = false;
        }    
    }         
    
    if (Blocked && RotatingBigVehicle())
    {
        Reason = "Rotating big vehicle";
        Blocked = false;
    }
       
    if (Blocked)
    {
        BlockedDist = dist;        
       
        ShowPath("Blocked by " $ blockedBy);                         
    }
   
    //SendDebugMessage("reason=" $ reason);
}

//called by SuggestStrafeDesination. returns location but should strafe to clear blockage.
function Vector ContinueToStrafe()
{
    local Controller C;
    local bool done;
    local float dist, hDist, variance, bDist;
    local vector loc, extent, ret;
      

    C = Controller;
    
    if (! Blocked)
    {
        return vect(0,0,0);
    }
       
    /*
    continue to strafe around blockage using previously calculated
    positioning. stop when clear of blockage.
    
    for horizontal blockages bot will continue to strafe in same direction
    until clear.
    
    for vertical blockages bot moves toward StrafeTarget.
    StrafeTarget is a place which can go to in order to clear the blockage.
    StrafeTarget is out beyond and a short distance above/below the edge of blockage.
    bot will first strafe out to StrafeTarget's horizontal location and then
    up/down to StrafeTarget.
    bot does not just move directly to StrafeTarget because in doing so bot can
    bump the blockage and be stopped.
    once bot around the edge of the blockage, bot can then safely start moving
    back towards target.
    */
    
    hDist = HorizontalDistanceBetweenPoints(C.Pawn.Location, Destination);
        
    done = false;

    extent = GetFrontalBlockageExtent();
                
    if (ZStrafing)
    {    
        //bot is moving out and up/down to vertical location to clear blockage
        
        //check if bot went far enough up/down to clear the blockage
        if (((C.Pawn.Location.Z - BlockedZ) * ZDirMult) > 25)
        {
            ShowPath("Cleared Z Block");          
            done = true;
        }
    }
    else if (! BackingUp)
    {
        //bot is trying to clear a horizontal blockage

        if (DrivingWheels())
        {
            //make wheeled vehicle need more to clear blockage so they drive
            //closer to the strafe location, otherwise they can turn right back
            //into the original blockage
            extent.X += 100;
            extent.Y += 100;
        }
        
        //trace out to see if blockage has been cleared            
        dist = BRGame.BlockedDist(self, BlockedDirection, hDist, C.Pawn.Location, extent);
        bDist = dist;
        if (dist == 0)
        {
            dist = hDist;
        }
        loc = C.Pawn.Location + (BlockedDirection * dist);

        //the mathematical law is that if you take the dot of a vector and a directional unit vector
        //you will get the length of the first vector along that direction, so if that length is longer
        //than our blockage dist, then we have cleared the blockage, as this location is closer along
        //the direction than the blockage dist
                    
        dist = BlockedDirection dot (loc - BlockedPawnLocation); 
   
        if ((dist - BlockedDist) > 25)
        {
            done = true;
            ShowPath("Clear blockage");
        }
        
        //check for wheeled vehicle stopped at frontal blockage. ut is generally
        //unable to resolve this so we must back the vehicle up.
        //stop the current strafe so that new one for backing up can be done.
        if ((! done) && DrivingWheels() && (vsize(C.Pawn.Velocity) < 50) && 
            (! ZBlocked) && (! RotatingBigVehicle()) && (bDist > 0) &&
            (bDist <= CollisionRadius))
        {
            done = true;
            ShowPath("Stalled at blockage");
        }
    }

    if ((WorldInfo.TimeSeconds - StrafeTime) > 10)
    {
        ShowPath("Strafe timed out");        
        done = true;
    }
    
    if (BackingUp && ((WorldInfo.TimeSeconds - StrafeTime) > 2))
    {
        ShowPath("Strafe timed out");        
        done = true;
    }
    
    if (ZStrafing && (! done) && (UTAirVehicle(C.Pawn) == none))
    {
        //we don't do this for flying vehicles. doesn't work well because when vehicle would 
        //go out to horizontal point out it also climbs
              
        if (HorizontalDistanceBetweenPoints(C.Pawn.Location, BlockedPawnLocation) > 
            HorizontalDistanceBetweenPoints(BlockedPawnLocation, StrafeTarget))
        {
            //bot is horizonally target when it's moved horizontally just a bit beyond it. 
            //now safe to start going directly up/down to target without bumping blockage. 
            HorizontallyNearTarget = true;
        } 
        
        if (! HorizontallyNearTarget)
        {
            //when going to a vertical strafe point to clear vertical blockage,
            //but first strafes horizontally to target.
            //once it reaches that point, then bot moves up or down to target
            //not doing this can result in bot hitting the blockage and getting knocked away,
            //especially if bot is jumping up trying to clear overhead blockage.
            
            ret = StrafeTarget;
            ret.Z = C.Pawn.Location.Z;
            
            return ret;
        }
        
        //example of this is bot being blocked above it's head but it
        //can't reach the strafe target because can't move upwards
        if (
            HorizontallyNearTarget &&
            (HorizontalDistanceBetweenPoints(C.Pawn.Location, BlockedPawnLocation) <
             HorizontalDistanceBetweenPoints(C.Pawn.Location, StrafeTarget))
           )
        {
            ShowPath("Going back to blocked position");
            done = true;         
        }
        
        //need to make sure while going out horizontally that we don't 
        //accidentally exceed the target, but we do want to test for 
        //exceeding target because bot could fall and not be able to reach
        //strafe point
        variance = 100;                     
    }    

    if (
        (DistanceBetweenPoints(C.Pawn.Location, BlockedPawnLocation) > 
         (DistanceBetweenPoints(BlockedPawnLocation, StrafeTarget) + variance)) ||
        (DistanceBetweenPoints(C.Pawn.Location, StrafeTarget) < 20)        
       )
    {
        ShowPath("Reached/Exceeded StrafeTarget");
        done = true;    
    }
            
    if (done)
    {
        TurnOffBlocking();
        return vect(0,0,0);
    }

    return StrafeTarget;
}

//return true if in big vehicle like tank and rotating it in place
function bool RotatingBigVehicle()
{
    if ((UTVehicle_Goliath(Bot.Pawn) != none) || (UTVehicle_Leviathan(Bot.Pawn) != none))
    {
        if ((WorldInfo.TimeSeconds - RotatingBigVehicleTime) <= 2)
        {
            return true;
        }
    }
    
    return false;
}

//return a trace extent to check for frontal blockages
function Vector GetFrontalBlockageExtent()
{
    local Vector extent;
    local float cRadius, cHeight;
         
    //it's especially important for wheeled vehicles to have a wide enough extent as 
    //we need lots of horizontal clearance to turn.
    //ut gives fluctuating and inaccurate values for radius and height, 
    //but for wheeled vehicles the radius ut reports will usually be
    //much larger than actual width of vehicle(because radius would be based on length of vehicle) 
    //so this should be enough.

    //we can't use this extent for search traces as since usually bigger than pawn, it would
    //make those traces fail when pawn's got a wall to it's side
    
    Controller.Pawn.GetBoundingCylinder(cRadius, cHeight);

    if (UTVehicle(Controller.Pawn) != none)
    {
        cRadius *= (4.0/5.0);
    }
        
    if (UTVehicle_Hoverboard(Controller.Pawn) != none)
    {
        cRadius = 12.5;
    }
       
    //must set both x & y, seems that they are alignned to world axis, not pawn rotation.
             
    extent.Y = cRadius;
    extent.X = cRadius;
    
    //using a small Z so trace is not blocked by variations in surface.
    //also, ut returns an inaccurate collision height so it's hard to calculate.
    extent.Z = 20;
    
    return extent;
}

//return an extent to performing trace searches for a strafe location
function Vector GetSearchTraceExtent()
{
    local float cRadius, cHeight;
    local Vector extent;
    
    //for these traces we must use an extent which is smaller in every way than
    //the pawn or else the trace will fail, getting blocked at zero dist if pawn
    //has a wall to it's side
    
    Controller.Pawn.GetBoundingCylinder(cRadius, cHeight);

    //important to create a collision which is no larger than core of pawn, else
    //is pawn is close to wall a trace will break because it collides with wall at 0 dist
       
    if (cRadius > cHeight)
    {   
        cRadius = cHeight;
    }
       
    if (UTVehicle_Hoverboard(Controller.Pawn) != none)
    {
        cRadius = 20;
    }

    //very frustrating. get real bad values for collision height. it fluctuates
    //and rarely matches the true collision height.
    //raptor for example is about 50 but ut reports it 60 - 90.
    //using 40 max for now, but may have to hard code for more vehicles.
    
    //need the height we actually use a littel smaller, as if vehicle is 
    //close to surface, we don't want horizontal traces colliding with surface. 
    if (cRadius > 40)
    {
        cRadius = 40;
    }
    
    extent.Z = cRadius;
    
    if ((Controller.Pawn.Physics == PHYS_Walking) ||
        (UTVehicle_Hoverboard(Controller.Pawn) != none))
    {
        //20 is definietly not enough to handle thin pillars which have wide bases.
        //25 would perhaps be enough.
        extent.Z = 30;
    }
    
    extent.X = cRadius;
    extent.Y = cRadius;
    
    return extent;       
}

function GetCollisionInfo()
{
    Controller.Pawn.GetBoundingCylinder(CollisionRadius, CollisionHeight);  
}

//check if bot is blocked and it so return a vector bot can strafe to in order to clear blockage.
//set Destination to bot's current target vector before calling.
function Vector SuggestStrafeDestination()
{
    local vector X, Y, Z;
    local Controller C;
    local vector dir, zDir, best, loc, extent, zExtent, ret, vtemp;
    local float zDist, hDist, depth;
    local Rotator rot;
    local float blockDist, dist;
    local float bestDistFromBot, bestClearance, targetDist, bestDist;
    local bool hBestNotBlocked, bestIsZLoc;
    local bool doSearch;

    bestIsZLoc = false;        
    C = Controller;

    GetCollisionInfo();  
    GetAxes(C.Pawn.Rotation,X,Y,Z);
    hDist = HorizontalDistanceBetweenPoints(C.Pawn.Location, Destination);   
    zDist = abs(Destination.Z - C.Pawn.Location.Z);
    
    ZDirMult = 1;
    if (Destination.Z < C.Pawn.Location.Z)
    {
        ZDirMult = -1;
    }

    zDir = vect(0,0,0);    
    zDir.Z = ZDirMult;
    
    
    if (Blocked)
    {
        //for debuging
        //spawn(class'UTEmit_VehicleHit',,,StrafeTarget);
    
        ret = ContinueToStrafe();
        if (Blocked)
        {
            return ret;
        }
        
        SoakAllowBlocking = false;
    }           

    if ((WorldInfo.TimeSeconds - BlockageTime) <= 0.50)
    {
        //this is an important optimization to keep blockage detection down to reasonable times
        return vect(0,0,0);    
    }
  
    CheckForVerticalBlockage();  
    if (! ZBLocked)
    {
        CheckForHorizontalBlockage();
        
        if (JumpOverBlockage)
        {
            return vect(0,0,0);
        }          
    }
        
    if (! Blocked)
    {
        return vect(0,0,0);
    }
        

    //
    //new blockage has been found. find way around it.
    //
    
    BlockageTime = WorldInfo.TimeSeconds;

    doSearch = true;                

    //check for wheeled vehicle stopped at frontal blockage. ut is generally
    //unable to resolve this so we must back the vehicle up.
    if (DrivingWheels() && (vsize(C.Pawn.Velocity) < 50) && 
        (! ZBlocked) && (! RotatingBigVehicle()) && (BlockedDist <= CollisionRadius))
    {
        doSearch = false;
        vtemp = Normal(vector(C.Pawn.Rotation));
        vtemp.z = 0;
        vtemp = normal(vtemp);
        best = C.Pawn.Location + ((vtemp * -1) * (CollisionRadius * 2));
        BackingUp = true;
        ShowPath("Backing vehicle up");
    }
    
    zDist = Destination.Z - C.Pawn.Location.Z;           
    rot = C.Pawn.Rotation;
    rot.Yaw = 0;
    loc = vect(0,0,0);
    targetDist = DistanceBetweenPoints(Destination, C.Pawn.Location);
    
    bestDist = 999999999;
    bestClearance = 0;
    bestDistFromBot = 999999999;
    
    BlockedPawnLocation = C.Pawn.Location;
    extent = GetSearchTraceExtent();
    
    //want a nice wide extent to clear the edge of a vertical blockage
    zExtent = GetFrontalBlockageExtent();

    /*
        trace around 360 degrees to find a good strafe location.
        
        each trace is also checked to see if bot could go out along that
        trace and then move up or down. this is used to clear vertical blockages.
        sometimes a horizontal blockage can totally block off bot and so bot can't
        just strafe to get by. instead bot can clear the vertical blockage (ceiling/floor)
        by moving out and up/down. 
    */ 
    while ((rot.Yaw <= 65536) && doSearch)
    {
        dir = vector(rot);
        
        //check at horizontal plane.
        //otherwise z path check traces can collide with surface.
        //vehicle could be oddly rotated but we still want to check horizontal plane.
        dir.Z = 0;
        dir = normal(dir);

        //
        //set best to a place bot can strafe horizontally to in order to clear blockage
        //               
                       
        blockDist = BRGame.BlockedDist(self, dir, 3000, C.Pawn.Location, extent);

        
        //if trace goes out beyond target then trim trace to target dist
        loc = C.Pawn.Location + (dir * blockDist);        
        dist = (loc - C.Pawn.Location) dot normal(Destination - C.Pawn.Location);
        if (dist > hDist)
        {
            blockDist = hDist;
        }
                            
                    
        if (blockDist == 0)
        {
            blockDist = hDist + 1;
        }
        else if (blockDist > 1)
        {
            //so that trace out from loc will not get blocked at 0 dist
            blockDist -= 1;
        }
              
        loc = C.Pawn.Location + (dir * blockDist);
        
        EvaluateStrafeLocation(loc, false, bestIsZLoc, best, bestDist, bestClearance, bestDistFromBot);
        if (best == loc)
        {
            bestIsZLoc = false;
        }
               
        if ((best == loc) && (bestClearance >= (targetDist - 25)))
        {
            hBestNotBlocked = true;
        }


        //
        //look for place that bot can use to clear a vertical blockage.
        //bot will first strafe out to that place's horizontal location, and then directly up/down to it.
        //
        
        //unless ZBlocked, will not look for vertical path if a horizontal path has good trace to target           
        if (
             ZBlocked || 
             ((! hBestNotBlocked) &&
             (abs(zDist) > 150) && 
             (zBlockDist > 0))
           )
        {       
             //trace out horizontally to find a break in surface
             
             depth = zBlockDist + 200;
             //if (depth > abs(zdist))
             //{
               //  depth = abs(zdist);
             //}
             
             depth = depth * (-1 * ZDirMult);
             
             CheckForZSurface(dir, 3000, depth, zExtent);
            
             if ((ZSurfaceBreakDist >= 0) && (BRGame.BlockedDist(self, dir, ZSurfaceBreakDist, C.Pawn.Location, extent) == 0))
             {            
                //found a surface break, now trace up/down through the breakage
                
                loc = C.Pawn.Location + (dir * ZSurfaceBreakDist);
                
                dist = BRGame.BlockedDist(self, zDir, abs(zDist), loc, extent);
                if (dist == 0)
                {
                    dist = abs(zDist);
                }
              
                dist = dist - (CollisionHeight + 10);
                dist = dist * (-1 * ZDirMult);             
                
                loc.Z = loc.Z - dist;
                if (Destination.Z > loc.Z)
                {
                    loc.Z = Destination.Z;
                }
                                               
                EvaluateStrafeLocation(loc, true, bestIsZLoc, best, bestDist, bestClearance, bestDistFromBot);
                if (best == loc)
                {
                    bestIsZLoc = true;
                }                
             }             
        }  
    
                
        rot.Yaw += 4096;
    }
  
    if (bestIsZloc) 
    {
        ShowPath("Using Z Strafe");
        ZStrafing = true;
    }
    
    HorizontallyNearTarget = false;
    StrafeTime = WorldInfo.TimeSeconds;
    StrafeTarget = best;
    
    StrafeDirection = normal(best - C.Pawn.Location);
    StrafeDirection.Z = 0;
    StrafeDirection = normal(StrafeDirection);
    SoakStrafeTargetDist = 999999999;    

    return ContinueToStrafe();
}

//return true if p's rotation or velocity is pointing general direction of targ location.
//ratio is -1 thru 1, 1 meaning pointing directly at targ, 0 pointing sideways, -1 pointing opposite direction.
//if opposite is true, then return true if pointing opposite direction to targ.
//if UseVelocity is true, then direction bot is moving is used instead of bot's rotation.
function bool IsInDirection(vector targ, optional float ratio, optional bool opposite, optional bool useVelocity)
{
    local vector LookDir, AimDir;

    if (UseVelocity)
    {
        LookDir = Bot.Pawn.Velocity;
    }
    else
    {
        LookDir = Vector(Controller.Pawn.Rotation);
    }
    
    LookDir.Z = 0;
    LookDir = Normal(LookDir);    
    
    AimDir = normal(targ - Controller.Pawn.Location);
    AimDir.Z = 0;
    AimDir = Normal(AimDir);

    //utpawn.NeedToTurn is ((LookDir Dot AimDir) < 0.93);
    //decrease ratio to get wider view
    
    if (ratio == 0)
    {
        ratio = 0.5;
    }
    
    if (opposite)
    {
        return ((LookDir Dot AimDir) <= (0 - ratio));
    }
     
    return ((LookDir Dot AimDir) >= ratio);        
}

//return true if bot is generally facing targ
function bool FacingGeneralDirection(vector targ, optional float ratio, optional bool opposite)
{
    return IsInDirection(targ, ratio, opposite);
}

//return true if p is facing opposite direction of targ
function bool FacingOppositeDirection(vector targ, optional float ratio)
{
    return FacingGeneralDirection(targ, ratio, true);
}

//return true if bot is going general direction of targ
function bool GoingGeneralDirection(vector targ, optional float ratio)
{
    return IsInDirection(targ, ratio, false, true);
}

//bot shoots ball from ball launcher
function ShootBall(optional float specialTossZ, optional bool getPassLock)
{
    PrimeLauncher();
    
    BallLauncher.SpecialTossZ = specialTossZ;
    
    BallLauncher.Activate();
    
    BallLauncher.LetBotShoot = true;
    
    if (getPassLock)
    {
        BallLauncher.GetPassLock();
    }
    
    BallLauncher.StartFire(0); //shoot ball
    
    //objective has changed so must reset cached vars
    SetVars();      
}

//get bot ready to fire ball.
//return true if bot is performing action to prime launcher for use.
function bool PrimeLauncher()
{ 
   if ((Bot == none) || (Bot.Pawn == none))
   {
       return false;
   }
   
   if (LeaveVehicle())
   {
       ShowPath("Leaving vehicle to shoot ball");
       
       //bot can get in loop entering and exiting vehicle unless we do this
       UTVehicle(Bot.Pawn).VehicleLostTime = WorldInfo.TimeSeconds + 5.0;
        
       return true;
   }
   
   if (Bot.Pawn.Weapon != BallLauncher)
   {
       ShowPath("Switch to ball launcher to shoot ball. Current weapon=" $ Bot.Pawn.Weapon);
          
       Bot.Pawn.SetActiveWeapon(BallLauncher);
       return true;
   }
   
   return false;
}

function bool LeaveVehicle()
{
    if (UTVehicle(Bot.Pawn) != none)
    {
        Bot.LeaveVehicle(true);
        return true;
    }
    
    return false;
}

//set playerManager vars needed by ai routines.
//optimizes cpu time and eliminates function(playerManager) type calls
//(single object parameter functions should be in that object but we want logic in ai class).
function SetVars()
{ 
    BRGame = UTBRGame(WorldInfo.Game);
    Squad = UTBRSquadAI(Bot.Squad);
    
    FormationCenter = Squad.FormationCenter(Bot);
    
    if (FormationCenter == none)
    {
        return;
    }
        
    FormationCenterLocation = FormationCenter.Location;
    FormationCenterTarget = FormationCenter;
    
    if (UTBRBall(FormationCenter) != none)
    {
        FormationCenterLocation = UTBRBall(FormationCenter).Position().Location;
        FormationCenterTarget = UTBRBall(FormationCenter).Position();
    }
    
    HorizontalDistanceFromFormationCenter =  HorizontalDistanceBetweenPoints(Bot.Pawn.Location, FormationCenterLocation);
    HorizontalDistanceFromEnemyGoal = HorizontalDistanceBetween(Bot.Pawn, Squad.EnemyGoal);
    DistanceFromEnemyGoal = DistanceBetween(Bot.Pawn, Squad.EnemyGoal);    
    DistanceFromFormationCenter = DistanceBetweenPoints(Bot.Pawn.Location, FormationCenterLocation); 
    MaxFormationCenterDefenseDistance = Squad.GetMaxDefenseDistanceFrom(FormationCenter, Bot);
    MaxFriendlyGoalDefenseDistance = Squad.GetMaxDefenseDistanceFrom(Squad.FriendlyGoal, Bot);
    VerticalDistanceFromFormationCenter = FormationCenterLocation.Z - Bot.Pawn.Location.Z;
    VerticalDistanceFromEnemyGoal = Squad.EnemyGoal.Location.Z - Bot.Pawn.Location.Z;
    HorizontalDistanceFromFriendlyGoal = HorizontalDistanceBetween(Bot.Pawn, Squad.FriendlyGoal);
    
    SetupSpecialPathAbilities();
    JumpHeight = Bot.MultiJumpZ + Bot.Pawn.JumpZ; 
         
    //horizontal distance can jump to max jump height
    JumpApexHorizontalDistance = (Bot.MultiJumpZ + Bot.Pawn.JumpZ) * (1 + (0.5 * UTBRGame(WorldInfo.Game).GetTravelGravityMultiplier()));
         
    BallResetDist = MaxFriendlyGoalDefenseDistance * 0.5;

    //radius for which a bot might try lobbing ball away from home goal    
    LobPassDistance = MaxFriendlyGoalDefenseDistance * (4.0/5.0);

    SpecialFocus = false;
           
    if (
         (UTBRGame(WorldInfo.Game).TestBotFocus != vect(0,0,0)) ||
         ShootingGoal || AttackBallBoosting || 
         DefenseBallBoosting || SpecialAiming ||
         (DoingBallReset && (GetBall(Bot.Pawn) != none))
       )
    {
        SpecialFocus = true;
    }
       
    FallbackPerimeterDistance = 1000;
    if (JumpApexHorizontalDistance > FallbackPerimeterDistance)
    {
        FallbackPerimeterDistance = JumpApexHorizontalDistance - 150;
        
        if (FallbackPerimeterDistance < 1000)
        {
            FallbackPerimeterDistance = 1000;
        }
        
        if (FallbackPerimeterDistance > 2000)
        {
            FallbackPerimeterDistance = 2000;
        }              
    }   
}

//based on function from UTBot, but with low grav and quad jump corrections factored in.
//must call this every ai tick to keep reseting the variables.
function SetupSpecialPathAbilities()
{
    local float gravMult, jumpMult, multiJumpBoost, multiJumpBoostMult, totalJumpZ;
    local int multiJump;

    if (BRGame.Test_NoSetupSpecialPathAbilities)
    {
        ShowPath("Not setting up SetupSpecialPathAbilities");
        return;
    }
    
    /*
       numbers from testing actual Z gains on jumps using default BR gravity of -420
       and 25 multijumpboost (unless otherwise specified)
       
       single jump 61.7
       double jump 133.37
       low grav single jump 259.20
       low grav double jump 560.20
       quad jump  273.64
       double jump using jump boots having
       775 multijumpboost   776.8
       0 multijumpboost double jump 122.85
       
       UTPawn.JumpZ is always 322.0
       This does not change in different gravity.
       Jump boots alter the UTPawn.MultiJumpBoost.

       based on these numbers, it would appear:
       -the gravity multiplier is just a straght division of gravity into a constant.
       -JumpZ multiplier is 0.1916
       MultiJumpBoost multiplier for large boost is 0.8438
       MultiJumpBoost multiplier for small boost is 0.4208

       The two boost numbers from above are the differences between using 25 or jump boots(775).
       Seems a large boost gets a better multiplier.       
       The actual Z gain seems to be difficult to calculate, perhaps based on some advanced physics
       stuff, so we can only try our best to estimate it.
       
       Based on these findings the calculations in UTBot class are grossly inaccurate,
       see UTBot.SpecialJumpTo and UTBot.SetupSpecialPathAbilities.
       It's too bad they can't use more staight forward calulations and also give us a good way to 
       know how high a bot can jump.
    */

    Bot.MultiJumpZ = 0;    
    Bot.MaxSpecialJumpZ = 0.f;    
              
    if (Bot.Pawn.bCanJump)
    {
        jumpMult = 0.1916;    
        multiJumpBoostMult = 0.8438;

        //normal ut3 grav is -520. lowgrav mutator is -100. BR default gravity is -420.    
        gravMult = (-420 / WorldInfo.WorldGravityZ);    
        multiJump = 0;        
        multiJumpBoost = 0;
        
        if (UTPawn(Bot.Pawn) != none)
        {
            multiJumpBoost = UTPawn(Bot.Pawn).MultiJumpBoost;
            
            if (UTPawn(Bot.Pawn).bCanDoubleJump)
            {
                multiJump = UTPawn(Bot.Pawn).MaxMultiJump;
            }
            
            if (UTPawn(Bot.Pawn).MultiJumpBoost <= 50)
            {
                //not sure where the cutoff is, so just using 50
                multiJumpBoostMult = 0.4208;
            }                        
        }    
        
        totalJumpZ = ((Bot.Pawn.JumpZ * (multiJump + 1)) * jumpMult);
        totalJumpZ =  totalJumpZ + (multiJumpBoost * multiJump * multiJumpBoostMult);
        totalJumpZ =  totalJumpZ * gravMult;
        
        //can't alter JumpZ, as that would probably affect actual jump height in game.
        //in UTBot they use forumla JumpZ + MultiJumpZ to arrive at total jump height.
        Bot.MultiJumpZ = totalJumpZ - Bot.Pawn.JumpZ;
        
        Bot.MaxSpecialJumpZ = totalJumpZ;
        
        //not doing anything to impactjumpz. don't have any test numbers on how it operates,
        //and not too concerned about it.
        if (Bot.bAllowedToImpactJump)
        {
            Bot.MaxSpecialJumpZ += Bot.ImpactJumpZ;
        }                           
    }
}

function bool IsLowGravity()
{
    return UTBRGame(WorldInfo.Game).IsLowGravity();
}

function UTBRBall GetBall(Actor actor)
{
    return UTBRGame(WorldInfo.Game).GetBall(actor);
}

//return a vector which can be used to move p dist away from goal.
//vector is perpendicular to goal face.
function vector MoveAwayFromGoalFaceVector(Pawn p, float dist, UTBRGoal goal)
{
    local vector temp1, temp2, startTrace, endTrace, hitLocation, hitNormal;
    local float hitDist1, hitDist2;
    local Actor actor;

    //we compare the vectors pointing in front and pointing behind goal at the pawn's Z
    //and use the one closest to the pawn
    
    temp1 = UTBRGame(WorldInfo.Game).PointAwayFromGoal(goal, dist, 1);   
    temp1.z = p.location.z;
    
    temp2 = UTBRGame(WorldInfo.Game).PointAwayFromGoal(goal, dist, -1);        
    temp2.z = p.location.z;

    startTrace = goal.Location;
    startTrace.Z = p.location.z;
    
    endTrace = temp1;
    hitDist1 = dist;
    foreach p.TraceActors(class'Actor', actor, hitLocation, hitNormal, endTrace, startTrace)
    {
        if (actor.bBlockActors)
        {
            hitDist1 = vsize(startTrace - hitLocation);
            break;
        }
    }
    
    endTrace = temp2;
    hitDist2 = dist;
    foreach p.TraceActors(class'Actor', actor, hitLocation, hitNormal, endTrace, startTrace)
    {
        if (actor.bBlockActors)
        {
            hitDist2 = vsize(startTrace - hitLocation);
            break;
        }
    }
    
    if (hitDist1 > hitDist2)
    {
        return temp1;
    }
    
    if (hitDist2 > hitDist1)
    {
       return temp2;
    }      
        
    if (vsize(temp1 - p.location) > vsize(temp2 - p.Location))
    {
       return temp2;
    }
       
    return temp1;   
}

/*
   return direction bot will aim to lob ball.
   
   ball flight path is calculated to detect how far ball will go and what ball will hit.
   unfortunately the calculations are rough and not 100% accurate and they are based on testing 
   benchmarks are are fairly close.
   using real world physics calculations don't seem like a good option because the ut3 engine isn't
   really real world physics and would differ from real world. 
   we really need some kind of ut3 call to calculate this for us to tell
   us where ut3 is going to put the ball but ut3 does not seem to provide anything. 
   the SPMA camera does calulate such flight path
   but unfortunately that's all hidden native code and hard wired into the camera and spma vehicle
   so we can't use it here.
*/
function Vector GetLobDirection()
{
    local vector dir, best, bestLandingPoint;
    local float max, min, bestDist, LobHeight, bestHeight, dist;
    local string s;
    local bool done;   
  
          
    if (IsLowGravity())
    {
        max = 250;        
    }
    else
    {
        max = 500;
    }
    
    LobHeight = max;
    
    min = 0;
    if (Controller.Pawn.Location.Z > Squad.EnemyGoal.Location.Z)
    {
        //if bot is above enemy goal then allow aiming down a little
        min = -150;
    }
    
    best = vect(0,0,0);
    bestDist = -1;
    done = false;
    
    //find a high point which is not blocked
    while ((LobHeight >= min) && (! done))
    {
        dir = vector(Controller.Pawn.Rotation);
        dir.z = 0;
        dir = normal(dir);
        dir = dir * 1000;
        dir.Z = dir.Z + LobHeight;
        dir = normal(dir);        
       
        s = "Failed";
        
        if (GoodLobDir(dir, LobHeight))
        {
            s = "Good";
            
            dist = LobBlockedDist;
            if (dist == 0)
            {
                dist = TheBall.TravelDistance(self, LobHeight);
            }
            
            if ((LobBlockedDist > 6000) || (LobBlockedDist == 0))
            {
                 done = true;
            }
            
            if (dist > bestDist)
            {
                best = dir;
                bestDist = dist;
                bestHeight = LobHeight;
                bestLandingPoint = LobLandingPoint;              
            }                        
        }
              
        //for debugging
        //ShowPath(Controller, "Lob Loop: " $ s $ "  Lob height " $ LobHeight $ " lobblockeddist=" $ LobBlockedDist $ " Ball Travel Dist = " $ TheBall.TravelDistance(playerManager, LobHeight) $ " reason " $ reason );
        if (s != "") {}  //stop compiler bitching about unused var
            
        LobHeight -= 25;
    }


    if (best != vect(0,0,0))
    {       
        dir = vector(Controller.Pawn.Rotation);
        dir.z = 0;
        dir = normal(dir);
        dir = dir * 1000;
        dir.Z = dir.Z + bestHeight;
        dir = normal(dir);
        best = dir;
        
        //for testing
        //spawn(class'UTEmit_VehicleHit',,,bestLandingPoint);
        if (bestLandingPoint != vect(0,0,0)) {}   //stop compiler from bitching about unused var
                    
        ShowPath("Selected Lob height = " $ bestHeight $ " Enemy Goal Dist = " $ HorizontalDistanceFromEnemyGoal);
    }
   
    return best;
}

//return true if dir is a good direction to lob ball 
function bool GoodLobDir(Vector dir, float height)
{
    local float dist, traceDist, traceUpDist, cap, travelDist, traceHeightMult;
    local vector dir2, startLoc, endLoc;
    local int i;
    local bool isBlocked;

    reason = "";
    
    /*
      find if ball path is blocked.
      do three traces in general shape of arc to get a somewhat realistic trace.
      trace one and three can be small because ball flight is mostly elliptical,
      that is, arc flatens out quickly and ball travels far before it goes down again.
      
                   trace two (needs to be as high as the ball flight path arc apex)
                 -----------------
     trace one /                   \  trace three
              /                     \
              
    */
    
    travelDist = TheBall.TravelDistance(self, height);

    isBlocked = false;
    startLoc = Controller.Pawn.GetPawnViewLocation();
    
    dir2 = dir;
    dir2.z = 0;
    dir2 = normal(dir2);     
    LobLandingPoint = startLoc + (dir2 * travelDist);
    
    //what fraction of travelDist for trace one and three. how high to lift trace two.
    traceHeightMult = 0.20;
    if (height >= 250)
    {
        //at higher aim height the arc is much higher so need a larger number
        traceHeightMult = 0.45;    
    }

    traceUpDist = travelDist * traceHeightMult; 
   
    
    for (i = 1; i <= 3; i++)
    {
        if (i == 1)
        {
           //first trace in direction gun is pointing
           dir2 = dir;
           traceDist = traceUpDist;
           if (height <= 0)
           {
               //for downward shots only do one straight trace
               traceDist = travelDist;
           }
        }
        
        if (i == 2)
        {
            //second trace horizontal direction gun is pointing
            dir2 = dir;
            dir2.z = 0;
            dir2 = normal(dir2);
            
            //this results in very accurate and even trace distances ending exactly at LobLandingPoint,
            //regardless of what traceHeightMult is set to.
            traceDist = travelDist - (HorizontalDistanceBetweenPoints(Controller.Pawn.GetPawnViewLocation(), startLoc) * 2);                       
        }
        
        if (i == 3)
        {
           //third trace to landing point
           dir2 = normal(LobLandingPoint - startLoc);
           traceDist = traceUpDist;
        }
                
        dist = 
            BRGame.BlockedDist(
               Controller.Pawn, 
               dir2, 
               traceDist,
               startLoc, 
               vect(30,30,30));

        if (dist > 0)
        {
            isBlocked = true;
        }
        else
        {
           dist = traceDist;
        }
        
        endLoc = startLoc + (dir2 * dist);
        
        //start for next trace
        startLoc = endLoc;            
      
        if (height <= 0)
        {
            //for downward shots only do one straight trace        
            break;
        }
    }
    
    LobLandingPoint = endLoc;
    
    LobBlockedDist = 0;
          
    if (isBlocked)
    {    
        LobBlockedDist = HorizontalDistanceBetweenPoints(Controller.Pawn.Location, LobLandingPoint);
    }

    if (LobBlockedDist > 0)
    {
        if ((height > 150))
        {
            //reason = "blocked at high aim height";
            //return false;
        }
         
        //allow blockages at certain distance if lob height is low.
        //don't want to lob high at blockage because can bounce all the way back.            
        if (LobBlockedDist <= 3000)
        {
            reason = "blocked by " $ BRGame.BlockingActor;
            return false;
        }
    }

    if (CheckTrace(CAN_SEE, Squad.EnemyGoal) && 
        MightOverShootOrHitGoal(height))
    {
        reason = "might overshoot enemygoal";
        return false;
    }               
       
    dir2 = dir;
    dir2.z = 0;
    dir2 = normal(dir2);

    //ball may bounce and go further, so check a reasonable distance out from landing point.
    //don't want to lob ball into an unstable area.
    dist = TheBall.TravelDistance(self) * 0.6;
    cap = 3000;
    if (dist > cap)
    {
        dist = cap;
    }
    
    if (! CheckLobLanding(dir2, dist, true))
    {
        return false;
    }
    
    //check in opposite direction. ball can hit wall and bounce back into killz.
    dist = dist - LobLandingCheckedDist;
    
    if (dist < 800)
    {
        dist = 800;
        
        if (LobLandingHeight > 2500)
        {
            //don't check backwards when high in the sky
            dist = -1;
        }        
    }

    cap = 1800;
    if (dist > cap)
    {
        dist = cap;
    }
       
    if (! CheckLobLanding(dir2 * -1, dist, false))
    {
        reason = reason $ "(checking backwards)";
        return false; 
    } 
       
    return true;
}

//return how far ball will fall as it travels over dist
function float GetFallDistance(float dist)
{
    return (dist / TheBall.GetTravelGravityMultiplier()) * (1300 / TheBall.GetShootSpeed());
}

//checks to make sure ball is lobbed into stable area, avoiding bad zones such as killz zones
//and falling out of path network.
//traces along dir up to maxDist until blocked looking for bad areas.
function bool CheckLobLanding(Vector dir, float maxDist, bool checkPathNetwork)
{
    local float dist, addedDist;
    local Vector v;
    local NavigationPoint n;
    local bool expandedMax;
    local int checkBadZoneFloor;
    
    LobLandingCheckedDist = 0;
    addedDist = 0;
    LobLandingHeight = 0;    
    checkBadZoneFloor = 0;

    while (addedDist <= maxDist) 
    {         
        LobLandingCheckedDist = addedDist;
                   
        v = dir;
        v.z = 0;
        v = normal(v);
        v = LobLandingPoint + (v * addedDist);
        
        //make sure we start trace high enough to find floor. 
        //if ball path is intersecting mesh then ball will usually bounce on top
        //of mesh so need to trace from up high to get top of mesh.
        v.z += 1000;  
        
        addedDist += 200;
                   
        dist = 10000;     
        dist = BRGame.BlockedDist(Controller.Pawn, vect(0,0,-1), dist, v, BallExtent());
        if (dist == 0)
        {
            //no floor found
            reason = "no floor";
            checkBadZoneFloor = 1;
            
            //rest of loop needs dist > 0
            continue;
        }
               
        if (dist > LobLandingHeight)
        {
            LobLandingHeight = dist;
        }
       
        v = v + (vect(0,0,-1) * dist);
        
        if (v.z <= WorldInfo.KillZ)
        {
            //lobbing into killz
            reason = "into global killz";
            checkBadZoneFloor = 1;

            //cap at killz
            v.z = WorldInfo.KillZ;
        }
        
        //estimate how high ball would be from ground at this point
        dist = LobLandingPoint.Z - v.Z;
        dist -= GetFallDistance(LobLandingCheckedDist);
        if (dist < 200)
        {
            //dist is used in below frontal blockage trace.
            //don't want small hills to stop trace, but do want big hills to stop,
            //as a big hill will usually stop ball from going out of bounds, but smaller
            //hills can get bounced over into out of bounds.        
            dist = 200;
        }

        if (checkBadZoneFloor > 0)
        {
            //encountered a bad zone and now have a floor reading to check how close we are to zone
            
            checkBadZoneFloor++;
            if (checkBadZoneFloor >= 4)
            {
                checkBadZoneFloor = 0;            
            }
                    
            if (dist <= 1000)
            {
                //ball is close to bad zone.
                //the height check is important because when ball is high in air it will easily pass over bad zone.
                //unfortunately can't check height when bad zone detected because the zone can have no floor so
                //must wait until we get a floor reading.
                return false;
            }
        } 
           
        if (checkPathNetwork)
        {    
            checkPathNetwork = false;
            
            foreach WorldInfo.RadiusNavigationPoints(class'NavigationPoint', n, v, 2000)
            {
                break;
            }        
            
            if (n == none)
            {
                //out of path network
                reason = "out of path network";
                return false;
            }
        }

        //trace from ground point forward to see if blockage.                
        v.z += dist;
        if (BRGame.BlockedDist(Controller.Pawn, dir, 200, v, BallExtent()) > 0)
        {
            //continue as long as not blocked.
            break;
        }
              
        if ((! expandedMax) && (addedDist >= (maxDist - 800)) && (dist > 500))
        {
            dist *= 1.5;
            
            if (dist > 4000) 
            {
                dist = 4000;
            }
            
            //ball is high in air and will still travel some distance so check out some more
            maxDist += dist;
            expandedMax = true;            
        }
    }
            
    if (checkBadZoneFloor > 0)
    {
        //end of traced path is near bad zone
        return false;
    }

    reason = "";
        
    return true;
}

/*
return true if ball shot in current direction and height might overshoot or hit enemy goal

don't hit or overshoot enemy goal on small maps.
target will be adjusted to make sure it falls below goal.
since the action is to lob ball, not score, 3 pointers are 
generally frowned on so a bot always doing it would be annoying.
*/
function bool MightOvershootOrHitGoal(float height)
{
    local float travelDist, overDist, adj;
    

    travelDist = HorizontalDistanceBetweenPoints(Bot.Pawn.Location, LobLandingPoint);

    //ball's projected landing point is not as far as goal, but is over goal Z,
    //and so ball can still travel further and fall into goal or overshoot 
    overDist = LobLandingPoint.z - Squad.EnemyGoal.Location.z;
    if ((travelDist < HorizontalDistanceFromEnemyGoal) && (overDist > 0) && (LobBlockedDist == 0))
    {
        if (GetFallDistance(HorizontalDistanceBetweenPoints(LobLandingPoint, Squad.EnemyGoal.Location)) < overDist)
        {
            return true;
        }
    }

    adj = 0;
       
    if (! Bot.Pawn.NeedToTurn(Squad.EnemyGoal.Location))
    {
        //bot is pointed at goal so keep ball away from goal
        adj = 100;
    }
    
    if ((height <= 0) && (LobLandingPoint.Z < Squad.EnemyGoal.Location.Z))
    {
       //for 0 or below heights we just do straight trace so ball will fall below LobLandingPoint so
       //as long as that's below goal it's ok.
       adj = 0;
    }    
        
    if (travelDist > (HorizontalDistanceFromEnemyGoal - adj))
    {
        return true;
    }
    
    return false;
}

//return extent for tracing with ball
function Vector BallExtent()
{
    return vect(30,30,30);
}

//return true if bot should be allowed to try jump to FormationCenter
function bool AllowJumpToFormationCenter()
{
    local float floorDist, goodDist;

    if (IsLowGravity())
    {
        return true;
    }
        
    floorDist = JumpApexHorizontalDistance * 4;
    if (floorDist > HorizontalDistanceFromFormationCenter)
    {
        floorDist = HorizontalDistanceFromFormationCenter;
    }    
   
    goodDist = CheckForFloorUpTo(FormationCenterLocation);
    
    return ((goodDist >= floorDist) || CanJumpTo(FormationCenter));
}

//return true if bot has a weapon which can heal node
function bool HasHealingWeapon(UTOnslaughtNodeObjective node)
{
    local UTWeapon w;
    local Pawn P;    
   
    P = Bot.Pawn;
	if (UTVehicle(Bot.Pawn) != None)
	{
	    P = UTVehicle(Bot.Pawn).Driver;
	}
	
	if (P == none)
	{
	    return false;
    }
    
    ForEach P.InvManager.InventoryActors(class'UTWeapon', w)
    {
        if (w.CanHeal(node))
        {
            return true;
        }
	}
	
	return false;
}

//from UTBot class
function bool NeedWeapon()
{
	local UTWeapon W;

    if ( Vehicle(Bot.Pawn) != None )
	{
	    //unfortunately can't evaluate weapons condition while in hoverboard,
	    //as weapons are swapped out so pawn weapon list only has towcable
		return false;
    }
		
	ForEach Bot.Pawn.InvManager.InventoryActors(class'UTWeapon',W)
	{
		if ( (W.AIRating > 0.5) && W.HasAmmo(0) )
			return false;
	}

	return true;
}

//return true if bot should get ball.
//even if FormationCenter is ball bot does not get ball if ball is held by bot's team, then bot covers the runner.
function bool ShouldPickupBall()
{
    local bool ret;

    ret =  (UTBRBall(FormationCenter) != none) && 
           (! UTBRBall(FormationCenter).IsHeldByTeamOf(Bot.Pawn)) &&
           (GetBall(Bot.Pawn) == none) &&
           (! GettingSpecialItem());
   
    return ret;
}
 
function bool GettingSpecialItem()
{
    return GettingWeapon || GettingVehicle || GettingSuperPickup;
}

function bool CheckFormationCenterChanged()
{
    local bool ret;
    
    if (LastFormationCenter != FormationCenter)
    {
        ShowPath("Formation center changed");
        
        LastFormationCenterLocation = FormationCenterLocation;
        SoakTarget = none;           
        FormationCenterReset();
        ret = true;
    }
    LastFormationCenter = FormationCenter;
    
    
    return ret;
}

function bool CheckFormationCenterMoved()
{
    local float diff;
    local bool ret;
    
    if ((WorldInfo.TimeSeconds - FormationCenterMovedTime) < 1.0)
    {
        //to keep cpu expense down, only allow the change to be detected once per second
        return false;
    }
    
    diff = vsize(CheckFormationCenterLocation - FormationCenterLocation);
    
    ret = false;
         
    if ((UTBRBall(FormationCenter) != none) && (diff > 400))
    {
        //bot was seeking defensive position because it couldn't reach ball, but now ball's
        //position has changed, so immediately try finding ball again
        ShowPath("Ball location changed.");
        FormationCenterReset();
        
        ret = true;
    } 
    else if (diff > 500)
    {
        //happens when bot is following a moving objective, such as covering the ball runner
        ShowPath("Formation center moved");
        FormationCenterReset();
        
        ret = true;
    }

    if (ret)
    { 
        FormationCenterMovedTime = WorldInfo.TimeSeconds;
        
        if (SpecialNavigating && IsFormationCenter(MoveTarget) && (! IsFormationCenter(PathTarget)))
        {
            //bot could be doing special move such as dodge to formation center,
            //so this will keep bot doing that. PathTarget could have been DefensivePosition,
            //which was just reset. 
            PathTarget = MoveTarget;
        }
        
        if (IsFormationCenter(PathTarget) && (MoveTarget != none) && 
            (! IsFormationCenter(MoveTarget)))
        {
            if ( 
                (DistanceBetween(Bot.Pawn, MoveTarget) > (DistanceFromFormationCenter + 25)) ||
                (! GoingGeneralDirection(PathTarget.Location, 0.1))
               )
            {
                //force a new path to be calculated if movetarget isn't getting bot closer to objective
                ForceRepath = true; 
            }
        }
    }
    
    return ret;
}

//event for when formation center is changed or moved
function FormationCenterReset()
{
    ForceCamp = false;
    CampAtDefensivePosition = false;
    ReachedFallback = false;
    DefensivePositionFailed = false;        
    ForceDefensivePosition = false;            
    Bot.DefensivePosition = none;
    SuppressDefensivePositionResetZ = 999999999;      
    CheckFormationCenterLocation = FormationCenterLocation;
    CachedDefensivePosition = none;
    FailPathTo = none;
    TriedDefensivePosition = false;
    SetPathSubTarget(none);
               
    CheckIfReachedFallback();    
}

//check if bot reached place near objective.
//special logic happens when bot is near objective, such as performing jumps when pathing fails.
function CheckIfReachedFallback()
{
    local bool ok;

    ok = false;
    
    if (ReachedFallBack)
    {
        CheckDefensivePositionReset();
        return;
    }
    
    if ((UTVehicle(Bot.Pawn) != none) && (HorizontalDistanceFromFormationCenter <= 2000))
    {
        ok = true;
    }  
    
    if (HorizontalDistanceFromFormationCenter <= FallbackPerimeterDistance)
    {
        ok = true;
    }      
    
    if (
        (Bot.DefensivePosition != none) && 
        (HorizontalDistanceFromFormationCenter < 
         HorizontalDistanceBetweenPoints(Bot.DefensivePosition.Location, FormationCenterLocation))
       )
    {
        //if bot is closer to objective than it is to DefensivePosition then consider it as having reached fallback.
        //can happen when bot can't path close to objective and so had to choose a place further out. 
        ok = true;
    }  
    
    if (ok && (! ReachedFallback))
    {
        //bot is directly going to objective but was taking an alternate path probably due to 
        //pathing failure. now that bot is close to objective, throw out the alternate path
        //and try seeking objective again. if new path attempt fails then bot will
        //attempt to resolve by moving directly to objective, using jumping for help if applicable.
         
        ShowPath("Entered formation center perimeter of " $ FormationCenter);
        SetReachedFallBack();
    }  
}

//set bot state to having reached fallback position.
//now bot's in position near to objective and can use special
//move to reach objective. because bot is now near to objective,
//it's expected to reach it in a timely manner.
function SetReachedFallBack()
{
    ReachedFallback = true;

    reason = "";
    CheckDefensivePositionReset();
    if (reason != "")
    {
        ShowPath(reason);
    }
}

//check and see if we should reset defensive position due to bot being in good
//position to head directly for objective
function CheckDefensivePositionReset()
{
    reason = "";
    
    if ((! SeekDirect) || (! ReachedFallback))
    {
        return;
    }
       
    if ((Bot.DefensivePosition != none) && SeekDirect && ReachedFallBack)
    {
        //we don't want to retask bot if already seeking objective directly.
        //retrying a path request in between path nodes might upset something.

        if ((! IsFormationCenter(PathTarget)) && 
            (PathTarget != Bot.DefensivePosition))
        {
            //bot on a side task such as getting weapon
            reason = "Not resetting defensive position because not currently seeking formation center";
        }
        else if (abs(VerticalDistanceFromFormationCenter) > SuppressDefensivePositionResetZ)
        {
            reason = "Not resetting defensive position because prior navigation failed";
        }    
        else if (SpecialNavigationNotGood(FormationCenter))    
        {
            //since we are routing to defensive position pathing to objective failed, and so 
            //normally you'd want to push to objective using special navigation once inside perimeter, 
            //but in this case the objective is not yet reachable so continue on the defensive position
            //and hopefully from there special navigation will work
            
            reason = "Not resetting defensive position: " $ reason;
        }
        else if ((VerticalDistanceFromFormationCenter < 50) &&
                 (HorizontalDistanceFromFormationCenter > 
                  HorizontalDistanceBetween(Bot.Pawn, Bot.DefensivePosition)))
        {
            //if objective is at or below bot's level then special navigation is most likely to work but
            //if defensive position is closer then might as well 
            //keep going to position just in case it helps in finding objective.

            reason = "Not resetting defensive position because it's closer than objective";
        }
        else if (Bot.Pawn.Physics == PHYS_Falling)
        {
            //rerouting mid air could have bad results, especially if bot was using jumppad
            reason = "Not resetting defensive position because falling";
        }
        else if ((UTBRBall(FormationCenter) != none) && (UTBRBall(FormationCenter).IsInMotion()))
        {
            reason = "Not resetting defensive position because ball moving";
        }
        else
        {
            //want bot to move to objective when it's close enough. continuing on to 
            //defensive position can be too much of a detour.
            //it's a tough call because in so many cases bot should immediately push to objective, and
            //in some cases going to defensive position might be the best way to reach objective, so
            //we try to detect which way is more likely to work.
            
            ShowPath("Reset defensive position inside perimeter.");
            Bot.DefensivePosition = none;
            
            TrySpecialNavigation(FormationCenterTarget, true);

            if (! SpecialNavigating)
            {          
                Retask();
            }
        }
    }   
}

function ClearSoakCycle()
{
    SoakSpecialNavigationTarget = none;
    ForcedJumpInSoak = false;
    IsSoaking = false; 
}

//stop bot from getting stuck and going nowhere. although ut seems to have anti soaking
//code, it doesn't seem to work. this routine helps make sure the bot keeps moving.
function bool CheckGeneralSoak(optional bool forceSoak)
{
    local float dist, diff, distVert;
    local bool ok, setDist, setDistVert;
     
    if ((Bot == none) || (Bot.Pawn == none) || (! CheckSoak) || (MoveTarget == none) || FindingRandomTarget)
    {
        SoakTarget = none;        
      
        return false;
    } 
      
    if (PerformingMeasuredJump() || TurningForGoalJumpShot)
    {
        //Measured Jump does it's own soak checks which measure how much closer bot 
        //is getting to target at each landing
        
        //we don't set SoakTarget to none because that reinitializes all soak vars.
        //we want the soak check to continue after the move is performed.
        //for example, a bot could be up against a wall and perform a special jump and still be stuck,
        //and we want to continue soak checks.
        
        SoakTime = WorldInfo.TimeSeconds;
        return false;
    }
      
    //since the timer is every second, substracted 0.5 second from each desired time    
    MaxSoakTime = 2.5;
    if ((UTVehicle_Goliath(Bot.Pawn) != none) || (UTVehicle_Leviathan(Bot.Pawn) != none))
    {
        MaxSoakTime = 3.5;
        if (RotatingBigVehicle())
        {
            MaxSoakTime = 4.5;
        }
        
        //when in big vehicle ut will sometimes change the movetarget on the fly without
        //notifying us! FindPathToObjective is not called and they probably just use the next
        //node in RouteCache. bot then moves on to the new target without us ever being notified. 
        //very annoying and seems like a very bad practice, and here we deal with it.
        
        if ((Bot.MoveTarget != MoveTarget) && (! SpecialNavigating))
        {
            ShowPath("UT changed movetarget to " $ Bot.MoveTarget $ " without calling FindPathToObjective! Allowing new target.");
            SetMoveTarget(Bot.MoveTarget);
        }        
    }
    
    if ((Bot.MoveTarget != MoveTarget) && (! SpecialNavigating))
    {
         //with tank code above, we must allow the change or tanks don't navigate so well.
         //it may happen mainly for tanks but have a warning here so the situation can be evaluated.
         //not so safe to allow this for other modes, as we need to be able to keep bot moving along it's 
         //path and detect for loops. if we allow ut to always change the movetarget without our knowing
         //then checking for loop conditions (something ut is quite prone to) and preventing them is not possible
         //or at least very difficult.
         ShowPath("Warning: UT changed movetarget to " $ Bot.MoveTarget $ " without calling FindPathToObjective! Ignoring.");
    }
        
    if (Bot.Pawn.Physics == PHYS_Walking)
    {
        MaxSoakTime = 1.5;
    }      

    //if moving target then evaluate soak on pawn position instead    
    diff = vsize(MoveTarget.Location) - vsize(SoakTargetLocation);
    if ((abs(diff) > 300) && (DistanceBetweenPoints(Bot.Pawn.Location, SoakLocation) > 300)) 
    {
        SoakTarget = none;
    }
    
    dist = DistanceBetweenPoints(Bot.Pawn.Location, MoveTarget.Location);
    distVert = abs(VerticalDistanceBetween(Bot.Pawn, MoveTarget));    
    
    ok = false;
    setDist = false;
    setDistVert = false;
        
    if (MoveTarget != SoakTarget)
    {
        ok = true;
        setDist = true;
        setDistVert = true;    
    }
    
    if (dist < (SoakBlockingDist - 25))
    {
        ok = true;
        setDist = true;    
    }

    if (distVert < (SoakBlockingDistVert - 25))
    {
        ok = true;
        setDistVert = true;    
    }
        
    if (ok)
    {
        //allow bot time to clear a blockage as long as they are getting closer.
        //SoakAllowBlocking is turned off by blocking routine when blockage cleared.
        
        if (setDist)
        {
            SoakBlockingDist = dist;
        }
        
        if (setDistVert)
        {
            SoakBlockingDistVert = distVert;
        }   
             
        SoakAllowBlocking = true;
        
        //soak situation totally cleared so can clear important vars tracking what
        //stage of soak we are in
        ClearSoakCycle();         
    }
    
    if (SoakAllowBlocking && Blocked)
    {
        //allow bot to navigate blockage as long as it's getting nearer to StrafeTarget
        
        if ((SoakStrafeTargetDist - 25) <= DistanceBetweenPoints(Bot.Pawn.Location, StrafeTarget))
        {
            SoakAllowBlocking = false;
        }
        
        SoakStrafeTargetDist = DistanceBetweenPoints(Bot.Pawn.Location, StrafeTarget);
    }
    
    
    if (SpecialNavigating)
    {
        //allows special navigation to be used to get out of a soaking spot.
        //gets reset when bot gets out of spot.
        //bot will first try to move along route by special navigating to MoveTarget,
        //and if that fails then abort route and try directly to PathTarget.
        SoakSpecialNavigationTarget = MoveTarget;
    }

    ok = false;
    setDist = false;
    setDistVert = false;
            
    if ((MoveTarget != SoakTarget) || (SoakAllowBlocking && Blocked))
    {
        ok = true;
        setDist = true;
        setDistVert = true;
    }
    
    if (dist < (SoakDist - 25))
    {
        ok = true;
        setDist = true;
    }
    
    if (distVert < (SoakDistVert - 25))
    {
        ok = true;
        setDistVert = true;
    }
         
    if (ok)
    {        
        //if don't manage dist and distVert separately then bot can be bouncing up and down forever
        if (setDist)
        {
            SoakDist = dist;
        }
        
        if (setDistVert)
        {
            SoakDistVert = distVert;
        }
            
        SoakTarget = MoveTarget;
        SoakTargetLocation = SoakTarget.Location;     
        SoakTime = WorldInfo.TimeSeconds;
        SoakLocation = Bot.Pawn.Location;     
    }
    
    if ((WorldInfo.TimeSeconds - SoakTime) >= MaxSoakTime)    
    {
        //bot is soaking

        SoakTime = WorldInfo.TimeSeconds;
        SoakAllowBlocking = true;
        IsSoaking = true;

        if (SpecialNavigating && CanInitiateJump() && (! ForcedJumpInSoak))
        {
            //kick it into measured jumping. sometimes bot can get stuck
            //because of some oddity of the terrain and no blockage is being,
            //detected, but a simple jump will get it out.
            
            if (SpecialNavigationPathTarget == none)
            {
                //not sure this could ever happen, but make sure forced navigation
                //is active so that PerformingMeasuredJump is turned on
                SpecialNavigationPathTarget = MoveTarget;
            }
            
            Reason = "Force jump due to soaking";
            ForceJump = true;
            ForcedJumpInSoak = true;
            EndSpecialNavigation();
            return true;
        }
                
        //before failing the path, try to keep on route and special navigate out 
        //of soaking spot if not yet tried so 
        if ((SoakSpecialNavigationTarget == none) && (MoveTarget != none) && (MoveTarget != PathTarget))
        {
            ShowPath("Trying special navigation to " $ MoveTarget $ " along route because soaking. UTBOT.MoveTarget=" $ Bot.MoveTarget);
                              
            if (TrySpecialNavigation(MoveTarget, true))
            {
                return true;
            }
        }  

        if ((UTVehicle(Bot.Pawn) != none) && UTVehicle(Bot.Pawn).bAllowedExit)
        {
            //bot leaves vehicle and continues current path on foot            

            ShowPath("Leaving vehicle because soaking");
            SoakTarget = none; //bot is starting fresh on foot so end the soak cycle
            ClearSoakCycle();             
            UTVehicle(Bot.Pawn).VehicleLostTime = WorldInfo.TimeSeconds + 10.0;           
            Bot.LeaveVehicle(true);           
                
            return true;            
        }
         
        
        //give up on path. make bot do something else
   
        ShowPath("Failing navigation due to soaking");
             
        if (ForcedSpecialNavigating())
        {
            Reason = "Soaking";
            EndSpecialNavigation(true);        
        }
        else
        {
            FailPath();
        }
        
        return true;
    }
    
    return false;
}

//cause FindPathToObjective to fail the path so that something else can be done
function FailPath()
{
    ShowPath("Failing path to " $ PathTarget);
    CheckPathSubTargetReset();
    FailPathTo = PathTarget;
    ForceRepath = true;
    Retask();
}

function SendDebugMessage(string msg)
{
    BRGame.SendDebugMessage(msg);
}

function StartBoostDefending(UTBRBall ball)
{ 
    ShowPath("Start boost defending");
    HasBoostWeapon(true);
    ball.TryingToDeflect = Bot;
    TheBall = ball;
    DefenseBallBoosting = true;
    SuppressBoostTime = WorldInfo.TimeSeconds + RandRange(0.0, 0.75); 
    SetBotTacticTimer();  
}

//return path dist from where bot is to RouteTarget of node
function float BRNodeRouteDist(UTBRPathNode node, RouteInfoStruct routeInfo)
{
    local float dist;
    
    dist = RouteDist(node);
    dist += node.GetPathDist(routeInfo);
    
    return dist; 
}

//return path dist to target
function float RouteDist(Actor routeTarget)
{
    return BRGame.RouteDist(Bot, routeTarget);    
}

//return true if bot is in between a1 and a2
function bool IsBetween(Actor a1, Actor a2)
{
    local float dist;
    
    dist = DistanceBetween(a1, a2);
    
    return
     (DistanceBetween(Bot.Pawn, a1) <= dist) &&
     (DistanceBetween(Bot.Pawn, a2) <= dist); 
}

//return route target for target which is the ultimate destination.
//used for UTBRPathNode searches.
function Actor GetRouteTarget(Actor target)
{
    if (target == Bot.DefensivePosition)
    {
        target = FormationCenter;
    }
    
    if (UTBRBall(target) != none)
    {
        if (UTBRBall(target).BHome) 
        {
            return UTBRBall(target).HomeBase;
        }
        
        return none;
    }
    
    if (UTVehicle(target) != none)
    {
        if ((UTVehicle(target).ParentFactory != none) &&
            (DistanceBetween(target, UTVehicle(target).ParentFactory) <= 50))
        {
            return UTVehicle(target).ParentFactory;
        }
        
        return none;
    }
       
    return target;
}

//checks for when bot enters the trigger radius of a br path node and fires retask if needed
function CheckBrNodeRadius()
{
    if ((UTBRPathNode(PathSubTarget) != none) &&
        (BRNodeRouteInfo.ReachedRadius != 0) &&
        (DistanceBetween(Bot.Pawn, PathSubTarget) <= BRNodeRouteInfo.ReachedRadius))
    {
        ShowPath("Retasking because entered ReachedRadius of " $ PathSubTarget);
        ForceRepath = true;
        Retask();
    }
}

//return true if bot can use weapons when has ball
function bool CanUseWeaponsWithBall()
{
     return (UTVehicle(Bot.Pawn) != none) || bool(UTBRGame(WorldInfo.Game).Settings.AllowBallCarrierWeapons);
}
   
replication
{
    if (Role == ROLE_Authority)
        SpectatorFreeCamera, PlayerReplicationInfo, TheBall, PawnLocation, PawnRotation, 
        PawnVisibleOnMiniMap, Kills, MaxMultiJump;
}

defaultproperties
{
   RemoteRole = ROLE_SimulatedProxy
   bAlwaysRelevant = true
   SpectatorFreeCamera = true
}
